/**
 * Accessibility JavaScript
 * 
 * Enhances keyboard navigation, focus management, and ARIA live regions
 * 
 * @package WCAG_Accessible_Pro
 * @since 1.0.0
 */

(function() {
    'use strict';

    /**
     * Initialize all accessibility features
     */
    function initAccessibility() {
        initKeyboardNavigation();
        initFocusManagement();
        initARIALiveRegions();
        initMobileMenu();
        initSkipLinks();
    }

    /**
     * Enhanced keyboard navigation
     * WCAG 2.1 SC 2.1.1 - Keyboard
     */
    function initKeyboardNavigation() {
        // Handle Enter and Space key on clickable elements
        document.addEventListener('keydown', function(e) {
            const target = e.target;
            
            // Check if element has role="button" or is a clickable div/span
            if (target.getAttribute('role') === 'button' || 
                target.classList.contains('clickable')) {
                
                // Enter or Space key
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    target.click();
                }
            }
        });

        // Trap focus in modals/dialogs
        const modals = document.querySelectorAll('[role="dialog"], [role="alertdialog"]');
        modals.forEach(function(modal) {
            trapFocus(modal);
        });
    }

    /**
     * Trap focus within a container
     * @param {HTMLElement} container - The container to trap focus in
     */
    function trapFocus(container) {
        const focusableElements = container.querySelectorAll(
            'a[href], button:not([disabled]), textarea:not([disabled]), ' +
            'input:not([disabled]), select:not([disabled]), [tabindex]:not([tabindex="-1"])'
        );
        
        if (focusableElements.length === 0) return;
        
        const firstFocusable = focusableElements[0];
        const lastFocusable = focusableElements[focusableElements.length - 1];
        
        container.addEventListener('keydown', function(e) {
            if (e.key !== 'Tab') return;
            
            if (e.shiftKey) {
                // Shift + Tab
                if (document.activeElement === firstFocusable) {
                    e.preventDefault();
                    lastFocusable.focus();
                }
            } else {
                // Tab
                if (document.activeElement === lastFocusable) {
                    e.preventDefault();
                    firstFocusable.focus();
                }
            }
        });
    }

    /**
     * Focus management
     * WCAG 2.1 SC 2.4.7 - Focus Visible
     */
    function initFocusManagement() {
        // Add visible focus indicator class
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Tab') {
                document.body.classList.add('user-is-tabbing');
            }
        });

        document.addEventListener('mousedown', function() {
            document.body.classList.remove('user-is-tabbing');
        });

        // Ensure focus is visible on all interactive elements
        const interactiveElements = document.querySelectorAll(
            'a, button, input, textarea, select, [tabindex]:not([tabindex="-1"])'
        );
        
        interactiveElements.forEach(function(element) {
            element.addEventListener('focus', function() {
                this.setAttribute('data-focus-visible', 'true');
            });
            
            element.addEventListener('blur', function() {
                this.removeAttribute('data-focus-visible');
            });
        });
    }

    /**
     * ARIA live regions for dynamic content
     * WCAG 2.1 SC 4.1.3 - Status Messages
     */
    function initARIALiveRegions() {
        // Create a live region for announcements
        const liveRegion = document.createElement('div');
        liveRegion.setAttribute('role', 'status');
        liveRegion.setAttribute('aria-live', 'polite');
        liveRegion.setAttribute('aria-atomic', 'true');
        liveRegion.className = 'sr-only';
        liveRegion.id = 'aria-live-region';
        document.body.appendChild(liveRegion);

        // Function to announce messages
        window.announceMessage = function(message) {
            const liveRegion = document.getElementById('aria-live-region');
            if (liveRegion) {
                liveRegion.textContent = message;
                
                // Clear after 5 seconds
                setTimeout(function() {
                    liveRegion.textContent = '';
                }, 5000);
            }
        };
    }

    /**
     * Mobile menu accessibility
     * WCAG 2.1 SC 1.4.13 - Content on Hover or Focus
     */
    function initMobileMenu() {
        const menuToggle = document.querySelector('.menu-toggle');
        const menu = document.querySelector('#primary-menu');
        
        if (!menuToggle || !menu) return;
        
        // Set initial ARIA attributes
        menuToggle.setAttribute('aria-expanded', 'false');
        menuToggle.setAttribute('aria-controls', 'primary-menu');
        menu.setAttribute('aria-hidden', 'true');
        
        // Toggle menu on click
        menuToggle.addEventListener('click', function() {
            const isExpanded = this.getAttribute('aria-expanded') === 'true';
            
            this.setAttribute('aria-expanded', !isExpanded);
            menu.setAttribute('aria-hidden', isExpanded);
            menu.classList.toggle('is-open');
            
            // Announce to screen readers
            if (window.announceMessage) {
                window.announceMessage(
                    isExpanded ? 'Menu closed' : 'Menu opened'
                );
            }
        });
        
        // Close menu on Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && menu.classList.contains('is-open')) {
                menuToggle.click();
                menuToggle.focus();
            }
        });
    }

    /**
     * Skip links functionality
     * WCAG 2.1 SC 2.4.1 - Bypass Blocks
     */
    function initSkipLinks() {
        const skipLinks = document.querySelectorAll('.skip-link');
        
        skipLinks.forEach(function(link) {
            link.addEventListener('click', function(e) {
                const targetId = this.getAttribute('href').substring(1);
                const target = document.getElementById(targetId);
                
                if (target) {
                    e.preventDefault();
                    
                    // Set focus to target
                    target.setAttribute('tabindex', '-1');
                    target.focus();
                    
                    // Scroll to target
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                    
                    // Remove tabindex after focus
                    target.addEventListener('blur', function() {
                        this.removeAttribute('tabindex');
                    }, { once: true });
                }
            });
        });
    }

    /**
     * Ensure all images have alt attributes
     * WCAG 2.1 SC 1.1.1 - Non-text Content
     */
    function checkImageAltText() {
        const images = document.querySelectorAll('img');
        
        images.forEach(function(img) {
            if (!img.hasAttribute('alt')) {
                console.warn('Image missing alt attribute:', img);
                // Add empty alt for decorative images
                img.setAttribute('alt', '');
            }
        });
    }

    /**
     * Initialize on DOM ready
     */
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initAccessibility);
    } else {
        initAccessibility();
    }

    // Check images after page load
    window.addEventListener('load', checkImageAltText);

})();

