<?php
/**
 * WCAG 2.1 AAA Enhancements
 * 
 * Additional WCAG features for 100% AAA compliance
 * - Autocomplete attributes (WCAG 1.3.6)
 * - Timeout warnings (WCAG 2.2.6)
 * - Form help system (WCAG 3.3.5)
 * - Heading hierarchy validation (WCAG 2.4.10)
 * - Breadcrumb navigation
 * - Accessibility toolbar (Font size, Dark mode, High contrast)
 *
 * @package WCAG_Accessible_Pro
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Add autocomplete attributes to form fields (WCAG 1.3.6 - Level AAA)
 */
function wcag_accessible_pro_add_autocomplete_attributes( $content ) {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_autocomplete' ) ) {
        return $content;
    }
    // Common autocomplete mappings
    $autocomplete_map = array(
        'email'     => 'email',
        'e-mail'    => 'email',
        'mail'      => 'email',
        'name'      => 'name',
        'fullname'  => 'name',
        'full-name' => 'name',
        'firstname' => 'given-name',
        'first-name'=> 'given-name',
        'lastname'  => 'family-name',
        'last-name' => 'family-name',
        'phone'     => 'tel',
        'telephone' => 'tel',
        'tel'       => 'tel',
        'mobile'    => 'tel',
        'address'   => 'street-address',
        'street'    => 'street-address',
        'city'      => 'address-level2',
        'state'     => 'address-level1',
        'zip'       => 'postal-code',
        'zipcode'   => 'postal-code',
        'postal'    => 'postal-code',
        'country'   => 'country-name',
        'company'   => 'organization',
        'organization' => 'organization',
        'website'   => 'url',
        'url'       => 'url',
    );
    
    foreach ( $autocomplete_map as $field_name => $autocomplete_value ) {
        // Match input fields with name or id containing the field name
        $pattern = '/<input([^>]*)(name|id)=["\']([^"\']*' . preg_quote( $field_name, '/' ) . '[^"\']*)["\']/i';
        
        if ( preg_match( $pattern, $content ) ) {
            // Add autocomplete attribute if not already present
            $content = preg_replace(
                '/(<input[^>]*(name|id)=["\'][^"\']*' . preg_quote( $field_name, '/' ) . '[^"\']*["\'][^>]*)(>)/i',
                '$1 autocomplete="' . esc_attr( $autocomplete_value ) . '"$3',
                $content
            );
        }
    }
    
    return $content;
}
add_filter( 'the_content', 'wcag_accessible_pro_add_autocomplete_attributes', 20 );

/**
 * Add autocomplete to Elementor forms (WCAG 1.3.6)
 */
function wcag_accessible_pro_elementor_form_autocomplete( $item, $item_index, $widget ) {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_autocomplete' ) ) {
        return $item;
    }

    $field_type = $item['field_type'];
    $field_label = strtolower( $item['field_label'] );
    
    $autocomplete_map = array(
        'email' => 'email',
        'name'  => 'name',
        'phone' => 'tel',
        'tel'   => 'tel',
    );
    
    // Check field type first
    if ( isset( $autocomplete_map[ $field_type ] ) ) {
        $item['autocomplete'] = $autocomplete_map[ $field_type ];
    }
    // Then check field label
    else {
        foreach ( $autocomplete_map as $keyword => $autocomplete_value ) {
            if ( strpos( $field_label, $keyword ) !== false ) {
                $item['autocomplete'] = $autocomplete_value;
                break;
            }
        }
    }
    
    return $item;
}
add_filter( 'elementor_pro/forms/render/item', 'wcag_accessible_pro_elementor_form_autocomplete', 15, 3 );

/**
 * Session timeout warning (WCAG 2.2.6 - Level AAA)
 */
function wcag_accessible_pro_timeout_warning() {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_timeout_warning' ) ) {
        return;
    }

    if ( ! is_user_logged_in() ) {
        return;
    }
    ?>
    <script>
    (function() {
        'use strict';
        
        // WordPress session timeout is typically 2 days (172800 seconds)
        // Warn 5 minutes before timeout
        const sessionTimeout = 172800000; // 48 hours in milliseconds
        const warningTime = 300000; // 5 minutes in milliseconds
        const warningTimeout = sessionTimeout - warningTime;
        
        let timeoutWarningTimer;
        let sessionTimeoutTimer;
        
        function showTimeoutWarning() {
            const liveRegion = document.getElementById('aria-live-region');
            if (liveRegion) {
                liveRegion.textContent = '<?php esc_html_e( 'Your session will expire in 5 minutes. Please save your work.', 'wcag-accessible-pro' ); ?>';
            }
            
            // Show visual warning
            const warning = document.createElement('div');
            warning.id = 'session-timeout-warning';
            warning.setAttribute('role', 'alert');
            warning.setAttribute('aria-live', 'assertive');
            warning.style.cssText = 'position:fixed;top:20px;right:20px;background:#fff3cd;color:#856404;padding:20px;border:2px solid #ffc107;border-radius:8px;box-shadow:0 4px 12px rgba(0,0,0,0.15);z-index:999999;max-width:400px;';
            warning.innerHTML = '<strong><?php esc_html_e( 'Session Timeout Warning', 'wcag-accessible-pro' ); ?></strong><p><?php esc_html_e( 'Your session will expire in 5 minutes. Please save your work or extend your session.', 'wcag-accessible-pro' ); ?></p><button id="extend-session" style="background:#007bff;color:#fff;border:none;padding:10px 20px;border-radius:4px;cursor:pointer;font-size:16px;min-height:44px;min-width:44px;"><?php esc_html_e( 'Extend Session', 'wcag-accessible-pro' ); ?></button>';
            
            document.body.appendChild(warning);
            
            // Focus on warning
            warning.setAttribute('tabindex', '-1');
            warning.focus();
            
            // Extend session button
            document.getElementById('extend-session').addEventListener('click', function() {
                // Trigger WordPress heartbeat to extend session
                if (typeof wp !== 'undefined' && wp.heartbeat) {
                    wp.heartbeat.connectNow();
                }
                
                // Remove warning
                warning.remove();
                
                // Reset timers
                clearTimeout(timeoutWarningTimer);
                clearTimeout(sessionTimeoutTimer);
                startTimeoutTimers();
            });
        }
        
        function startTimeoutTimers() {
            timeoutWarningTimer = setTimeout(showTimeoutWarning, warningTimeout);
        }
        
        // Start timers on page load
        startTimeoutTimers();
        
        // Reset timers on user activity
        let activityTimeout;
        function resetOnActivity() {
            clearTimeout(activityTimeout);
            activityTimeout = setTimeout(function() {
                clearTimeout(timeoutWarningTimer);
                clearTimeout(sessionTimeoutTimer);
                startTimeoutTimers();
            }, 1000);
        }
        
        document.addEventListener('mousemove', resetOnActivity);
        document.addEventListener('keydown', resetOnActivity);
        document.addEventListener('click', resetOnActivity);
        
    })();
    </script>
    <?php
}
add_action( 'wp_footer', 'wcag_accessible_pro_timeout_warning' );

/**
 * Heading hierarchy validation (WCAG 2.4.10 - Level AAA)
 */
function wcag_accessible_pro_heading_validation() {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_heading_validation' ) ) {
        return;
    }

    if ( ! current_user_can( 'edit_posts' ) ) {
        return;
    }
    ?>
    <script>
    (function() {
        'use strict';
        
        function validateHeadingHierarchy() {
            const headings = document.querySelectorAll('h1, h2, h3, h4, h5, h6');
            let previousLevel = 0;
            let errors = [];
            
            headings.forEach(function(heading, index) {
                const currentLevel = parseInt(heading.tagName.substring(1));
                
                // Check if heading level is skipped
                if (currentLevel - previousLevel > 1) {
                    errors.push({
                        element: heading,
                        message: 'Heading level skipped from H' + previousLevel + ' to H' + currentLevel
                    });
                }
                
                previousLevel = currentLevel;
            });
            
            // Show errors in console for editors
            if (errors.length > 0) {
                console.group('⚠️ WCAG 2.4.10 - Heading Hierarchy Issues');
                errors.forEach(function(error) {
                    console.warn(error.message, error.element);
                });
                console.groupEnd();
            }
        }
        
        // Run validation after page load
        window.addEventListener('load', validateHeadingHierarchy);
        
    })();
    </script>
    <?php
}
add_action( 'wp_footer', 'wcag_accessible_pro_heading_validation' );

/**
 * Accessibility Toolbar (Font size, Dark mode, High contrast)
 */
function wcag_accessible_pro_accessibility_toolbar() {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_accessibility_toolbar' ) ) {
        return;
    }
    ?>
    <div id="wcag-accessibility-toolbar" class="wcag-toolbar" role="toolbar" aria-label="<?php esc_attr_e( 'Accessibility Options', 'wcag-accessible-pro' ); ?>">
        <button id="wcag-font-decrease" class="wcag-toolbar-button" aria-label="<?php esc_attr_e( 'Decrease font size', 'wcag-accessible-pro' ); ?>" title="<?php esc_attr_e( 'Decrease font size', 'wcag-accessible-pro' ); ?>">A-</button>
        <button id="wcag-font-increase" class="wcag-toolbar-button" aria-label="<?php esc_attr_e( 'Increase font size', 'wcag-accessible-pro' ); ?>" title="<?php esc_attr_e( 'Increase font size', 'wcag-accessible-pro' ); ?>">A+</button>
        <button id="wcag-dark-mode" class="wcag-toolbar-button" aria-label="<?php esc_attr_e( 'Toggle dark mode', 'wcag-accessible-pro' ); ?>" title="<?php esc_attr_e( 'Toggle dark mode', 'wcag-accessible-pro' ); ?>">🌙</button>
        <button id="wcag-high-contrast" class="wcag-toolbar-button" aria-label="<?php esc_attr_e( 'Toggle high contrast', 'wcag-accessible-pro' ); ?>" title="<?php esc_attr_e( 'Toggle high contrast', 'wcag-accessible-pro' ); ?>">◐</button>
        <button id="wcag-reset" class="wcag-toolbar-button" aria-label="<?php esc_attr_e( 'Reset accessibility settings', 'wcag-accessible-pro' ); ?>" title="<?php esc_attr_e( 'Reset accessibility settings', 'wcag-accessible-pro' ); ?>">↺</button>
    </div>
    
    <style>
    .wcag-toolbar {
        position: fixed !important;
        top: 50% !important;
        right: 0 !important;
        transform: translateY(-50%) !important;
        background: #ffffff !important;
        border: 2px solid #333333 !important;
        border-right: none !important;
        border-radius: 8px 0 0 8px !important;
        padding: 10px 5px !important;
        display: flex !important;
        flex-direction: column !important;
        gap: 8px !important;
        z-index: 999998 !important;
        box-shadow: -2px 0 8px rgba(0,0,0,0.1) !important;
    }
    
    .wcag-toolbar-button {
        min-width: 44px !important;
        min-height: 44px !important;
        background: #f8f9fa !important;
        border: 2px solid #333333 !important;
        border-radius: 4px !important;
        font-size: 18px !important;
        font-weight: 700 !important;
        cursor: pointer !important;
        transition: all 0.2s ease !important;
        color: #1a1a1a !important;
    }
    
    .wcag-toolbar-button:hover {
        background: #e9ecef !important;
        transform: scale(1.05) !important;
    }
    
    .wcag-toolbar-button:focus {
        outline: 3px solid #0066cc !important;
        outline-offset: 2px !important;
        box-shadow: 0 0 0 4px rgba(0, 102, 204, 0.2) !important;
    }
    
    .wcag-toolbar-button:active {
        transform: scale(0.95) !important;
    }
    
    /* Dark mode styles */
    body.wcag-dark-mode {
        background-color: #1a1a1a !important;
        color: #ffffff !important;
    }

    body.wcag-dark-mode h1,
    body.wcag-dark-mode h2,
    body.wcag-dark-mode h3,
    body.wcag-dark-mode h4,
    body.wcag-dark-mode h5,
    body.wcag-dark-mode h6 {
        color: #ffffff !important;
    }

    body.wcag-dark-mode p,
    body.wcag-dark-mode span,
    body.wcag-dark-mode div,
    body.wcag-dark-mode li {
        color: #ffffff !important;
    }

    body.wcag-dark-mode a {
        color: #66b3ff !important;
    }

    body.wcag-dark-mode .wcag-toolbar {
        background: #2a2a2a !important;
        border-color: #ffffff !important;
    }

    body.wcag-dark-mode .wcag-toolbar-button {
        background: #3a3a3a !important;
        color: #ffffff !important;
        border-color: #ffffff !important;
    }
    
    /* High contrast mode */
    body.wcag-high-contrast {
        background-color: #000000 !important;
        color: #ffff00 !important;
    }

    body.wcag-high-contrast h1,
    body.wcag-high-contrast h2,
    body.wcag-high-contrast h3,
    body.wcag-high-contrast h4,
    body.wcag-high-contrast h5,
    body.wcag-high-contrast h6 {
        color: #ffff00 !important;
        background-color: transparent !important;
    }

    body.wcag-high-contrast p,
    body.wcag-high-contrast span,
    body.wcag-high-contrast div,
    body.wcag-high-contrast li {
        color: #ffff00 !important;
    }

    body.wcag-high-contrast a {
        color: #00ffff !important;
        text-decoration: underline !important;
    }

    body.wcag-high-contrast button,
    body.wcag-high-contrast input[type="submit"] {
        background: #ffff00 !important;
        color: #000000 !important;
        border: 3px solid #ffffff !important;
    }

    body.wcag-high-contrast .wcag-toolbar {
        background: #000000 !important;
        border-color: #ffff00 !important;
    }

    body.wcag-high-contrast .wcag-toolbar-button {
        background: #000000 !important;
        color: #ffff00 !important;
        border-color: #ffff00 !important;
    }
    
    @media (prefers-reduced-motion: reduce) {
        .wcag-toolbar-button {
            transition: none !important;
        }
        
        .wcag-toolbar-button:hover,
        .wcag-toolbar-button:active {
            transform: none !important;
        }
    }
    </style>
    
    <script>
    (function() {
        'use strict';

        let fontSize = 100;

        // Apply font size to html element
        function applyFontSize(size) {
            // Set on html element (affects rem units throughout the page)
            document.documentElement.style.fontSize = size + '%';

            // Also set a CSS custom property for advanced usage
            document.documentElement.style.setProperty('--wcag-font-size', size + '%');

            // Update aria-live region for screen readers
            announceChange('<?php esc_html_e( 'Font size is now', 'wcag-accessible-pro' ); ?> ' + size + '%');
        }

        // Load saved preferences on page load
        if (localStorage.getItem('wcag-font-size')) {
            fontSize = parseInt(localStorage.getItem('wcag-font-size'));
            applyFontSize(fontSize);
        }

        if (localStorage.getItem('wcag-dark-mode') === 'true') {
            document.body.classList.add('wcag-dark-mode');
        }

        if (localStorage.getItem('wcag-high-contrast') === 'true') {
            document.body.classList.add('wcag-high-contrast');
        }

        // Font size decrease
        document.getElementById('wcag-font-decrease').addEventListener('click', function() {
            if (fontSize > 80) {
                fontSize -= 10;
                applyFontSize(fontSize);
                localStorage.setItem('wcag-font-size', fontSize);
                announceChange('<?php esc_html_e( 'Font size decreased to', 'wcag-accessible-pro' ); ?> ' + fontSize + '%');
            } else {
                announceChange('<?php esc_html_e( 'Minimum font size reached', 'wcag-accessible-pro' ); ?>');
            }
        });

        // Font size increase
        document.getElementById('wcag-font-increase').addEventListener('click', function() {
            if (fontSize < 200) {
                fontSize += 10;
                applyFontSize(fontSize);
                localStorage.setItem('wcag-font-size', fontSize);
                announceChange('<?php esc_html_e( 'Font size increased to', 'wcag-accessible-pro' ); ?> ' + fontSize + '%');
            } else {
                announceChange('<?php esc_html_e( 'Maximum font size reached', 'wcag-accessible-pro' ); ?>');
            }
        });
        
        // Dark mode toggle
        document.getElementById('wcag-dark-mode').addEventListener('click', function() {
            document.body.classList.toggle('wcag-dark-mode');
            const isDark = document.body.classList.contains('wcag-dark-mode');
            localStorage.setItem('wcag-dark-mode', isDark);
            announceChange(isDark ? '<?php esc_html_e( 'Dark mode enabled', 'wcag-accessible-pro' ); ?>' : '<?php esc_html_e( 'Dark mode disabled', 'wcag-accessible-pro' ); ?>');
        });
        
        // High contrast toggle
        document.getElementById('wcag-high-contrast').addEventListener('click', function() {
            document.body.classList.toggle('wcag-high-contrast');
            const isHighContrast = document.body.classList.contains('wcag-high-contrast');
            localStorage.setItem('wcag-high-contrast', isHighContrast);
            announceChange(isHighContrast ? '<?php esc_html_e( 'High contrast enabled', 'wcag-accessible-pro' ); ?>' : '<?php esc_html_e( 'High contrast disabled', 'wcag-accessible-pro' ); ?>');
        });
        
        // Reset all
        document.getElementById('wcag-reset').addEventListener('click', function() {
            fontSize = 100;
            applyFontSize(fontSize);
            document.body.classList.remove('wcag-dark-mode', 'wcag-high-contrast');
            localStorage.removeItem('wcag-font-size');
            localStorage.removeItem('wcag-dark-mode');
            localStorage.removeItem('wcag-high-contrast');
            announceChange('<?php esc_html_e( 'Accessibility settings reset to default', 'wcag-accessible-pro' ); ?>');
        });
        
        function announceChange(message) {
            if (typeof window.announceMessage === 'function') {
                window.announceMessage(message);
            } else {
                // Fallback: Create temporary aria-live region
                const liveRegion = document.getElementById('aria-live-region') || createLiveRegion();
                liveRegion.textContent = message;

                // Clear after announcement
                setTimeout(function() {
                    liveRegion.textContent = '';
                }, 1000);
            }
        }

        function createLiveRegion() {
            const region = document.createElement('div');
            region.id = 'aria-live-region';
            region.setAttribute('role', 'status');
            region.setAttribute('aria-live', 'polite');
            region.setAttribute('aria-atomic', 'true');
            region.style.cssText = 'position:absolute;left:-10000px;width:1px;height:1px;overflow:hidden;';
            document.body.appendChild(region);
            return region;
        }

        // Keyboard support for toolbar buttons (Enter and Space)
        document.querySelectorAll('.wcag-toolbar-button').forEach(function(button) {
            button.addEventListener('keydown', function(e) {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    this.click();
                }
            });
        });

    })();
    </script>
    <?php
}
add_action( 'wp_footer', 'wcag_accessible_pro_accessibility_toolbar', 5 );

