<?php
/**
 * Video & Audio Accessibility Features
 * 
 * WCAG 1.2 - Time-based Media Compliance
 * - 1.2.1 Audio-only and Video-only (A)
 * - 1.2.2 Captions (Prerecorded) (A)
 * - 1.2.3 Audio Description or Media Alternative (A)
 * - 1.2.4 Captions (Live) (AA)
 * - 1.2.5 Audio Description (Prerecorded) (AA)
 * - 1.2.8 Media Alternative (Prerecorded) (AAA)
 *
 * @package WCAG_Accessible_Pro
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Enhance video embeds with accessibility features
 */
function wcag_accessible_pro_enhance_video_embeds( $html, $url, $attr, $post_id ) {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_video_accessibility' ) ) {
        return $html;
    }

    // Check if it's a video embed
    if ( strpos( $html, '<iframe' ) === false ) {
        return $html;
    }
    
    // Add title attribute if missing (WCAG 4.1.2)
    if ( strpos( $html, 'title=' ) === false ) {
        $title = 'Video embed';
        
        // Try to extract title from URL
        if ( strpos( $url, 'youtube.com' ) !== false || strpos( $url, 'youtu.be' ) !== false ) {
            $title = 'YouTube video';
        } elseif ( strpos( $url, 'vimeo.com' ) !== false ) {
            $title = 'Vimeo video';
        }
        
        $html = str_replace( '<iframe', '<iframe title="' . esc_attr( $title ) . '"', $html );
    }
    
    // Wrap in figure with caption for transcript link
    $wrapper = '<figure class="video-embed-wrapper">';
    $wrapper .= $html;
    $wrapper .= '<figcaption class="video-caption">';
    $wrapper .= '<p class="video-accessibility-notice">';
    $wrapper .= '<strong>' . esc_html__( 'Accessibility:', 'wcag-accessible-pro' ) . '</strong> ';
    $wrapper .= esc_html__( 'This video should include captions and audio description. If you need a transcript, please contact us.', 'wcag-accessible-pro' );
    $wrapper .= '</p>';
    $wrapper .= '</figcaption>';
    $wrapper .= '</figure>';
    
    return $wrapper;
}
add_filter( 'embed_oembed_html', 'wcag_accessible_pro_enhance_video_embeds', 10, 4 );

/**
 * Add accessibility controls to HTML5 video elements
 */
function wcag_accessible_pro_enhance_html5_video( $content ) {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_video_accessibility' ) ) {
        return $content;
    }

    // Find all <video> tags
    if ( strpos( $content, '<video' ) === false ) {
        return $content;
    }
    
    // Add controls attribute if missing
    $content = preg_replace(
        '/<video(?![^>]*controls)/',
        '<video controls',
        $content
    );
    
    // Add preload="metadata" for better accessibility
    $content = preg_replace(
        '/<video(?![^>]*preload)/',
        '<video preload="metadata"',
        $content
    );
    
    // Check if captions track exists
    if ( strpos( $content, '<track' ) === false ) {
        // Add notice about missing captions
        $content = preg_replace(
            '/<\/video>/',
            '</video><p class="video-accessibility-warning" role="alert"><strong>' . esc_html__( 'Accessibility Notice:', 'wcag-accessible-pro' ) . '</strong> ' . esc_html__( 'This video is missing captions. Please add a caption track for WCAG compliance.', 'wcag-accessible-pro' ) . '</p>',
            $content
        );
    }
    
    return $content;
}
add_filter( 'the_content', 'wcag_accessible_pro_enhance_html5_video', 15 );

/**
 * Elementor Video Widget Accessibility Enhancement
 */
function wcag_accessible_pro_elementor_video_accessibility( $content, $widget ) {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_video_accessibility' ) ) {
        return $content;
    }

    if ( 'video' !== $widget->get_name() ) {
        return $content;
    }
    
    $settings = $widget->get_settings();
    
    // Add accessibility notice
    $notice = '<div class="elementor-video-accessibility-notice" style="margin-top:15px;padding:15px;background:#f8f9fa;border-left:4px solid #0066cc;border-radius:4px;">';
    $notice .= '<p style="margin:0;font-size:14px;line-height:1.6;">';
    $notice .= '<strong>' . esc_html__( 'Video Accessibility Checklist:', 'wcag-accessible-pro' ) . '</strong><br>';
    $notice .= '✓ ' . esc_html__( 'Captions for all audio content', 'wcag-accessible-pro' ) . '<br>';
    $notice .= '✓ ' . esc_html__( 'Audio description for visual content', 'wcag-accessible-pro' ) . '<br>';
    $notice .= '✓ ' . esc_html__( 'Transcript available on request', 'wcag-accessible-pro' );
    $notice .= '</p>';
    $notice .= '</div>';
    
    return $content . $notice;
}
add_filter( 'elementor/widget/render_content', 'wcag_accessible_pro_elementor_video_accessibility', 10, 2 );

/**
 * Add video accessibility admin notice
 */
function wcag_accessible_pro_video_admin_notice() {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_video_accessibility' ) ) {
        return;
    }

    $screen = get_current_screen();

    if ( ! $screen || ( $screen->base !== 'post' && $screen->base !== 'post-new' ) ) {
        return;
    }
    ?>
    <div class="notice notice-info">
        <h3><?php esc_html_e( 'WCAG 2.1 Video Accessibility Requirements', 'wcag-accessible-pro' ); ?></h3>
        <p><strong><?php esc_html_e( 'When adding videos to your content, ensure:', 'wcag-accessible-pro' ); ?></strong></p>
        <ul style="list-style:disc;margin-left:20px;">
            <li><strong><?php esc_html_e( 'Captions (WCAG 1.2.2 - Level A):', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'All prerecorded audio content must have synchronized captions.', 'wcag-accessible-pro' ); ?></li>
            <li><strong><?php esc_html_e( 'Audio Description (WCAG 1.2.5 - Level AA):', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Provide audio description for all prerecorded video content.', 'wcag-accessible-pro' ); ?></li>
            <li><strong><?php esc_html_e( 'Transcript (WCAG 1.2.8 - Level AAA):', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Provide a full text transcript for all media.', 'wcag-accessible-pro' ); ?></li>
        </ul>
        <p><strong><?php esc_html_e( 'How to add captions to HTML5 video:', 'wcag-accessible-pro' ); ?></strong></p>
        <pre style="background:#f5f5f5;padding:15px;border-radius:4px;overflow-x:auto;"><code>&lt;video controls&gt;
  &lt;source src="video.mp4" type="video/mp4"&gt;
  &lt;track kind="captions" src="captions-de.vtt" srclang="de" label="Deutsch" default&gt;
  &lt;track kind="captions" src="captions-en.vtt" srclang="en" label="English"&gt;
  &lt;track kind="descriptions" src="descriptions.vtt" srclang="de" label="Audio Description"&gt;
&lt;/video&gt;</code></pre>
        <p><strong><?php esc_html_e( 'Resources:', 'wcag-accessible-pro' ); ?></strong></p>
        <ul style="list-style:disc;margin-left:20px;">
            <li><a href="https://www.w3.org/WAI/media/av/" target="_blank" rel="noopener"><?php esc_html_e( 'W3C: Making Audio and Video Media Accessible', 'wcag-accessible-pro' ); ?></a></li>
            <li><a href="https://www.youtube.com/accessibility" target="_blank" rel="noopener"><?php esc_html_e( 'YouTube: Add captions to your videos', 'wcag-accessible-pro' ); ?></a></li>
            <li><a href="https://support.google.com/youtube/answer/2734796" target="_blank" rel="noopener"><?php esc_html_e( 'YouTube: Auto-generated captions', 'wcag-accessible-pro' ); ?></a></li>
        </ul>
    </div>
    <?php
}
add_action( 'admin_notices', 'wcag_accessible_pro_video_admin_notice' );

/**
 * Video accessibility styles
 */
function wcag_accessible_pro_video_styles() {
    ?>
    <style>
    .video-embed-wrapper {
        margin: 30px 0 !important;
        padding: 0 !important;
    }
    
    .video-embed-wrapper iframe {
        max-width: 100% !important;
        height: auto !important;
        aspect-ratio: 16 / 9 !important;
    }
    
    .video-caption {
        margin-top: 15px !important;
        padding: 15px !important;
        background: #f8f9fa !important;
        border-left: 4px solid #0066cc !important;
        border-radius: 4px !important;
    }
    
    .video-accessibility-notice {
        margin: 0 !important;
        font-size: 14px !important;
        line-height: 1.6 !important;
        color: #495057 !important;
    }
    
    .video-accessibility-warning {
        margin: 15px 0 !important;
        padding: 15px !important;
        background: #fff3cd !important;
        border-left: 4px solid #ffc107 !important;
        border-radius: 4px !important;
        color: #856404 !important;
        font-size: 14px !important;
        line-height: 1.6 !important;
    }
    
    /* Ensure video controls are accessible */
    video::-webkit-media-controls-panel {
        background-color: rgba(0, 0, 0, 0.8) !important;
    }
    
    video::-webkit-media-controls-play-button,
    video::-webkit-media-controls-mute-button,
    video::-webkit-media-controls-fullscreen-button {
        min-width: 44px !important;
        min-height: 44px !important;
    }
    
    /* Focus styles for video controls */
    video:focus {
        outline: 3px solid #0066cc !important;
        outline-offset: 2px !important;
    }
    
    /* Responsive video embeds */
    @media (max-width: 768px) {
        .video-embed-wrapper iframe {
            width: 100% !important;
            height: auto !important;
        }
    }
    </style>
    <?php
}
add_action( 'wp_head', 'wcag_accessible_pro_video_styles' );

/**
 * Create WebVTT caption file template
 */
function wcag_accessible_pro_create_vtt_template() {
    if ( ! current_user_can( 'edit_posts' ) ) {
        return;
    }
    
    if ( isset( $_GET['wcag_create_vtt_template'] ) && check_admin_referer( 'wcag_vtt_template' ) ) {
        $vtt_content = "WEBVTT\n\n";
        $vtt_content .= "NOTE\n";
        $vtt_content .= "This is a WebVTT caption file template.\n";
        $vtt_content .= "Replace the timestamps and text with your actual captions.\n\n";
        $vtt_content .= "00:00:00.000 --> 00:00:05.000\n";
        $vtt_content .= "First caption text goes here.\n\n";
        $vtt_content .= "00:00:05.000 --> 00:00:10.000\n";
        $vtt_content .= "Second caption text goes here.\n\n";
        $vtt_content .= "00:00:10.000 --> 00:00:15.000\n";
        $vtt_content .= "Third caption text goes here.\n";
        
        header( 'Content-Type: text/vtt; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename="captions-template.vtt"' );
        echo $vtt_content;
        exit;
    }
}
add_action( 'admin_init', 'wcag_accessible_pro_create_vtt_template' );

/**
 * Add VTT template download link to media library
 */
function wcag_accessible_pro_vtt_template_link() {
    $screen = get_current_screen();
    
    if ( ! $screen || $screen->base !== 'upload' ) {
        return;
    }
    
    $url = wp_nonce_url( admin_url( '?wcag_create_vtt_template=1' ), 'wcag_vtt_template' );
    ?>
    <div class="notice notice-info">
        <p>
            <strong><?php esc_html_e( 'Need to add captions to your videos?', 'wcag-accessible-pro' ); ?></strong>
            <a href="<?php echo esc_url( $url ); ?>" class="button button-primary" style="margin-left:10px;">
                <?php esc_html_e( 'Download WebVTT Caption Template', 'wcag-accessible-pro' ); ?>
            </a>
        </p>
    </div>
    <?php
}
add_action( 'admin_notices', 'wcag_accessible_pro_vtt_template_link' );

