<?php
/**
 * Breadcrumb Navigation (WCAG 2.4.8 - Level AAA)
 * 
 * Provides multiple ways to locate content
 * Includes Schema.org BreadcrumbList markup for SEO
 *
 * @package WCAG_Accessible_Pro
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Display breadcrumb navigation
 *
 * @param array $args Breadcrumb arguments
 */
function wcag_accessible_pro_breadcrumbs( $args = array() ) {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_breadcrumbs' ) ) {
        return;
    }

    // Default arguments
    $defaults = array(
        'home_label'    => __( 'Home', 'wcag-accessible-pro' ),
        'separator'     => '<span class="breadcrumb-separator" aria-hidden="true"> / </span>',
        'show_current'  => true,
        'show_on_home'  => false,
        'schema_markup' => true,
    );
    
    $args = wp_parse_args( $args, $defaults );
    
    // Don't display on homepage unless specified
    if ( is_front_page() && ! $args['show_on_home'] ) {
        return;
    }
    
    global $post;
    
    $breadcrumbs = array();
    $position = 1;
    
    // Home link
    $breadcrumbs[] = array(
        'url'      => home_url( '/' ),
        'label'    => $args['home_label'],
        'position' => $position++,
    );
    
    // Build breadcrumb trail
    if ( is_category() ) {
        $category = get_queried_object();
        
        // Parent categories
        if ( $category->parent ) {
            $parent_categories = array();
            $parent_id = $category->parent;
            
            while ( $parent_id ) {
                $parent = get_category( $parent_id );
                $parent_categories[] = array(
                    'url'      => get_category_link( $parent->term_id ),
                    'label'    => $parent->name,
                    'position' => $position++,
                );
                $parent_id = $parent->parent;
            }
            
            $breadcrumbs = array_merge( $breadcrumbs, array_reverse( $parent_categories ) );
        }
        
        // Current category
        $breadcrumbs[] = array(
            'url'      => '',
            'label'    => $category->name,
            'position' => $position++,
            'current'  => true,
        );
        
    } elseif ( is_tag() ) {
        $tag = get_queried_object();
        $breadcrumbs[] = array(
            'url'      => '',
            'label'    => sprintf( __( 'Tag: %s', 'wcag-accessible-pro' ), $tag->name ),
            'position' => $position++,
            'current'  => true,
        );
        
    } elseif ( is_author() ) {
        $author = get_queried_object();
        $breadcrumbs[] = array(
            'url'      => '',
            'label'    => sprintf( __( 'Author: %s', 'wcag-accessible-pro' ), $author->display_name ),
            'position' => $position++,
            'current'  => true,
        );
        
    } elseif ( is_date() ) {
        if ( is_day() ) {
            $breadcrumbs[] = array(
                'url'      => get_year_link( get_the_time( 'Y' ) ),
                'label'    => get_the_time( 'Y' ),
                'position' => $position++,
            );
            $breadcrumbs[] = array(
                'url'      => get_month_link( get_the_time( 'Y' ), get_the_time( 'm' ) ),
                'label'    => get_the_time( 'F' ),
                'position' => $position++,
            );
            $breadcrumbs[] = array(
                'url'      => '',
                'label'    => get_the_time( 'd' ),
                'position' => $position++,
                'current'  => true,
            );
        } elseif ( is_month() ) {
            $breadcrumbs[] = array(
                'url'      => get_year_link( get_the_time( 'Y' ) ),
                'label'    => get_the_time( 'Y' ),
                'position' => $position++,
            );
            $breadcrumbs[] = array(
                'url'      => '',
                'label'    => get_the_time( 'F' ),
                'position' => $position++,
                'current'  => true,
            );
        } elseif ( is_year() ) {
            $breadcrumbs[] = array(
                'url'      => '',
                'label'    => get_the_time( 'Y' ),
                'position' => $position++,
                'current'  => true,
            );
        }
        
    } elseif ( is_search() ) {
        $breadcrumbs[] = array(
            'url'      => '',
            'label'    => sprintf( __( 'Search Results for: %s', 'wcag-accessible-pro' ), get_search_query() ),
            'position' => $position++,
            'current'  => true,
        );
        
    } elseif ( is_404() ) {
        $breadcrumbs[] = array(
            'url'      => '',
            'label'    => __( '404 Not Found', 'wcag-accessible-pro' ),
            'position' => $position++,
            'current'  => true,
        );
        
    } elseif ( is_single() ) {
        // Post type
        $post_type = get_post_type();
        
        if ( $post_type === 'post' ) {
            // Categories
            $categories = get_the_category();
            if ( $categories ) {
                $category = $categories[0];
                
                // Parent categories
                if ( $category->parent ) {
                    $parent_categories = array();
                    $parent_id = $category->parent;
                    
                    while ( $parent_id ) {
                        $parent = get_category( $parent_id );
                        $parent_categories[] = array(
                            'url'      => get_category_link( $parent->term_id ),
                            'label'    => $parent->name,
                            'position' => $position++,
                        );
                        $parent_id = $parent->parent;
                    }
                    
                    $breadcrumbs = array_merge( $breadcrumbs, array_reverse( $parent_categories ) );
                }
                
                $breadcrumbs[] = array(
                    'url'      => get_category_link( $category->term_id ),
                    'label'    => $category->name,
                    'position' => $position++,
                );
            }
        } else {
            // Custom post type archive
            $post_type_object = get_post_type_object( $post_type );
            if ( $post_type_object->has_archive ) {
                $breadcrumbs[] = array(
                    'url'      => get_post_type_archive_link( $post_type ),
                    'label'    => $post_type_object->labels->name,
                    'position' => $position++,
                );
            }
        }
        
        // Current post
        $breadcrumbs[] = array(
            'url'      => '',
            'label'    => get_the_title(),
            'position' => $position++,
            'current'  => true,
        );
        
    } elseif ( is_page() ) {
        // Parent pages
        if ( $post->post_parent ) {
            $parent_pages = array();
            $parent_id = $post->post_parent;
            
            while ( $parent_id ) {
                $parent = get_post( $parent_id );
                $parent_pages[] = array(
                    'url'      => get_permalink( $parent->ID ),
                    'label'    => get_the_title( $parent->ID ),
                    'position' => $position++,
                );
                $parent_id = $parent->post_parent;
            }
            
            $breadcrumbs = array_merge( $breadcrumbs, array_reverse( $parent_pages ) );
        }
        
        // Current page
        $breadcrumbs[] = array(
            'url'      => '',
            'label'    => get_the_title(),
            'position' => $position++,
            'current'  => true,
        );
        
    } elseif ( is_post_type_archive() ) {
        $post_type_object = get_queried_object();
        $breadcrumbs[] = array(
            'url'      => '',
            'label'    => $post_type_object->labels->name,
            'position' => $position++,
            'current'  => true,
        );
    }
    
    // Output breadcrumbs
    if ( count( $breadcrumbs ) > 1 ) {
        echo '<nav class="breadcrumb-navigation" aria-label="' . esc_attr__( 'Breadcrumb', 'wcag-accessible-pro' ) . '">';
        
        if ( $args['schema_markup'] ) {
            echo '<ol class="breadcrumbs" itemscope itemtype="https://schema.org/BreadcrumbList">';
        } else {
            echo '<ol class="breadcrumbs">';
        }
        
        foreach ( $breadcrumbs as $crumb ) {
            $is_current = isset( $crumb['current'] ) && $crumb['current'];
            
            if ( $args['schema_markup'] ) {
                echo '<li class="breadcrumb-item' . ( $is_current ? ' breadcrumb-current' : '' ) . '" itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">';
            } else {
                echo '<li class="breadcrumb-item' . ( $is_current ? ' breadcrumb-current' : '' ) . '">';
            }
            
            if ( ! empty( $crumb['url'] ) ) {
                if ( $args['schema_markup'] ) {
                    echo '<a href="' . esc_url( $crumb['url'] ) . '" itemprop="item">';
                    echo '<span itemprop="name">' . esc_html( $crumb['label'] ) . '</span>';
                    echo '</a>';
                    echo '<meta itemprop="position" content="' . esc_attr( $crumb['position'] ) . '">';
                } else {
                    echo '<a href="' . esc_url( $crumb['url'] ) . '">' . esc_html( $crumb['label'] ) . '</a>';
                }
                
                if ( $crumb !== end( $breadcrumbs ) ) {
                    echo $args['separator'];
                }
            } else {
                if ( $args['schema_markup'] ) {
                    echo '<span itemprop="name" aria-current="page">' . esc_html( $crumb['label'] ) . '</span>';
                    echo '<meta itemprop="position" content="' . esc_attr( $crumb['position'] ) . '">';
                } else {
                    echo '<span aria-current="page">' . esc_html( $crumb['label'] ) . '</span>';
                }
            }
            
            echo '</li>';
        }
        
        echo '</ol>';
        echo '</nav>';
    }
}

/**
 * Add breadcrumb styles
 */
function wcag_accessible_pro_breadcrumb_styles() {
    ?>
    <style>
    .breadcrumb-navigation {
        margin: 20px 0 !important;
        padding: 15px 20px !important;
        background: #f8f9fa !important;
        border-radius: 4px !important;
    }
    
    .breadcrumbs {
        list-style: none !important;
        display: flex !important;
        flex-wrap: wrap !important;
        align-items: center !important;
        gap: 8px !important;
        margin: 0 !important;
        padding: 0 !important;
    }
    
    .breadcrumb-item {
        display: inline-flex !important;
        align-items: center !important;
        font-size: 14px !important;
        line-height: 1.5 !important;
    }
    
    .breadcrumb-item a {
        color: #0066cc !important;
        text-decoration: none !important;
        transition: color 0.2s ease !important;
    }
    
    .breadcrumb-item a:hover {
        color: #004499 !important;
        text-decoration: underline !important;
    }
    
    .breadcrumb-item a:focus {
        outline: 3px solid #0066cc !important;
        outline-offset: 2px !important;
        border-radius: 2px !important;
    }
    
    .breadcrumb-current {
        color: #6c757d !important;
        font-weight: 600 !important;
    }
    
    .breadcrumb-separator {
        margin: 0 4px !important;
        color: #6c757d !important;
    }
    
    @media (prefers-reduced-motion: reduce) {
        .breadcrumb-item a {
            transition: none !important;
        }
    }
    </style>
    <?php
}
add_action( 'wp_head', 'wcag_accessible_pro_breadcrumb_styles' );

