<?php
/**
 * WCAG Theme Settings Panel
 * 
 * Admin-Einstellungen zum Aktivieren/Deaktivieren einzelner WCAG-Features
 *
 * @package WCAG_Accessible_Pro
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Add admin menu
 */
function wcag_accessible_pro_add_admin_menu() {
    add_theme_page(
        __( 'WCAG Accessibility Settings', 'wcag-accessible-pro' ),
        __( 'WCAG Settings', 'wcag-accessible-pro' ),
        'manage_options',
        'wcag-accessibility-settings',
        'wcag_accessible_pro_settings_page'
    );
}
add_action( 'admin_menu', 'wcag_accessible_pro_add_admin_menu' );

/**
 * Register settings
 */
function wcag_accessible_pro_settings_init() {
    register_setting( 'wcag_accessibility', 'wcag_accessibility_options' );
    
    // General Settings Section
    add_settings_section(
        'wcag_general_section',
        __( 'Feature Activation', 'wcag-accessible-pro' ),
        'wcag_accessible_pro_general_section_callback',
        'wcag_accessibility'
    );
    
    // Breadcrumb Navigation
    add_settings_field(
        'enable_breadcrumbs',
        __( 'Breadcrumb Navigation', 'wcag-accessible-pro' ),
        'wcag_accessible_pro_checkbox_field',
        'wcag_accessibility',
        'wcag_general_section',
        array(
            'label_for'   => 'enable_breadcrumbs',
            'description' => __( 'Enable breadcrumb navigation (WCAG 2.4.8 - Level AAA)', 'wcag-accessible-pro' ),
            'default'     => true,
        )
    );
    
    // Accessibility Toolbar
    add_settings_field(
        'enable_accessibility_toolbar',
        __( 'Accessibility Toolbar', 'wcag-accessible-pro' ),
        'wcag_accessible_pro_checkbox_field',
        'wcag_accessibility',
        'wcag_general_section',
        array(
            'label_for'   => 'enable_accessibility_toolbar',
            'description' => __( 'Enable floating accessibility toolbar (Font size, Dark mode, High contrast)', 'wcag-accessible-pro' ),
            'default'     => true,
        )
    );
    
    // Session Timeout Warning
    add_settings_field(
        'enable_timeout_warning',
        __( 'Session Timeout Warning', 'wcag-accessible-pro' ),
        'wcag_accessible_pro_checkbox_field',
        'wcag_accessibility',
        'wcag_general_section',
        array(
            'label_for'   => 'enable_timeout_warning',
            'description' => __( 'Enable session timeout warning (WCAG 2.2.6 - Level AAA)', 'wcag-accessible-pro' ),
            'default'     => true,
        )
    );
    
    // Autocomplete Attributes
    add_settings_field(
        'enable_autocomplete',
        __( 'Autocomplete Attributes', 'wcag-accessible-pro' ),
        'wcag_accessible_pro_checkbox_field',
        'wcag_accessibility',
        'wcag_general_section',
        array(
            'label_for'   => 'enable_autocomplete',
            'description' => __( 'Automatically add autocomplete attributes to form fields (WCAG 1.3.6 - Level AAA)', 'wcag-accessible-pro' ),
            'default'     => true,
        )
    );
    
    // Video Accessibility
    add_settings_field(
        'enable_video_accessibility',
        __( 'Video Accessibility Enhancements', 'wcag-accessible-pro' ),
        'wcag_accessible_pro_checkbox_field',
        'wcag_accessibility',
        'wcag_general_section',
        array(
            'label_for'   => 'enable_video_accessibility',
            'description' => __( 'Enable video/audio accessibility features (WCAG 1.2.x)', 'wcag-accessible-pro' ),
            'default'     => true,
        )
    );
    
    // Heading Hierarchy Validation
    add_settings_field(
        'enable_heading_validation',
        __( 'Heading Hierarchy Validation', 'wcag-accessible-pro' ),
        'wcag_accessible_pro_checkbox_field',
        'wcag_accessibility',
        'wcag_general_section',
        array(
            'label_for'   => 'enable_heading_validation',
            'description' => __( 'Enable heading hierarchy validation for editors (WCAG 2.4.10 - Level AAA)', 'wcag-accessible-pro' ),
            'default'     => true,
        )
    );
    
    // Skip Links
    add_settings_field(
        'enable_skip_links',
        __( 'Skip Links', 'wcag-accessible-pro' ),
        'wcag_accessible_pro_checkbox_field',
        'wcag_accessibility',
        'wcag_general_section',
        array(
            'label_for'   => 'enable_skip_links',
            'description' => __( 'Enable skip to content links (WCAG 2.4.1 - Level A)', 'wcag-accessible-pro' ),
            'default'     => true,
        )
    );
    
    // Elementor Enhancements
    add_settings_field(
        'enable_elementor_enhancements',
        __( 'Elementor WCAG Enhancements', 'wcag-accessible-pro' ),
        'wcag_accessible_pro_checkbox_field',
        'wcag_accessibility',
        'wcag_general_section',
        array(
            'label_for'   => 'enable_elementor_enhancements',
            'description' => __( 'Enable Elementor-specific WCAG enhancements (semantic wrappers, ARIA, etc.)', 'wcag-accessible-pro' ),
            'default'     => true,
        )
    );
    
    // Reduced Motion Support
    add_settings_field(
        'enable_reduced_motion',
        __( 'Reduced Motion Support', 'wcag-accessible-pro' ),
        'wcag_accessible_pro_checkbox_field',
        'wcag_accessibility',
        'wcag_general_section',
        array(
            'label_for'   => 'enable_reduced_motion',
            'description' => __( 'Respect user\'s reduced motion preference (WCAG 2.3.3 - Level AAA)', 'wcag-accessible-pro' ),
            'default'     => true,
        )
    );
    
    // Link Text Validation
    add_settings_field(
        'enable_link_validation',
        __( 'Link Text Validation', 'wcag-accessible-pro' ),
        'wcag_accessible_pro_checkbox_field',
        'wcag_accessibility',
        'wcag_general_section',
        array(
            'label_for'   => 'enable_link_validation',
            'description' => __( 'Enhance generic link texts with screen reader context (WCAG 2.4.9 - Level AAA)', 'wcag-accessible-pro' ),
            'default'     => true,
        )
    );
    
    // Image Figure/Figcaption
    add_settings_field(
        'enable_image_figures',
        __( 'Semantic Image Figures', 'wcag-accessible-pro' ),
        'wcag_accessible_pro_checkbox_field',
        'wcag_accessibility',
        'wcag_general_section',
        array(
            'label_for'   => 'enable_image_figures',
            'description' => __( 'Wrap images with captions in semantic <figure> tags', 'wcag-accessible-pro' ),
            'default'     => true,
        )
    );
}
add_action( 'admin_init', 'wcag_accessible_pro_settings_init' );

/**
 * Section callback
 */
function wcag_accessible_pro_general_section_callback() {
    echo '<p>' . esc_html__( 'Aktivieren oder deaktivieren Sie einzelne WCAG-Features. Deaktivieren Sie Features, die Sie bereits manuell implementiert haben.', 'wcag-accessible-pro' ) . '</p>';
}

/**
 * Checkbox field callback
 */
function wcag_accessible_pro_checkbox_field( $args ) {
    $options = get_option( 'wcag_accessibility_options' );
    $value   = isset( $options[ $args['label_for'] ] ) ? $options[ $args['label_for'] ] : ( isset( $args['default'] ) ? $args['default'] : false );
    ?>
    <label>
        <input 
            type="checkbox" 
            id="<?php echo esc_attr( $args['label_for'] ); ?>" 
            name="wcag_accessibility_options[<?php echo esc_attr( $args['label_for'] ); ?>]" 
            value="1" 
            <?php checked( $value, 1 ); ?>
        >
        <?php echo esc_html( $args['description'] ); ?>
    </label>
    <?php
}

/**
 * Settings page HTML
 */
function wcag_accessible_pro_settings_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    
    // Save message
    if ( isset( $_GET['settings-updated'] ) ) {
        add_settings_error(
            'wcag_accessibility_messages',
            'wcag_accessibility_message',
            __( 'Settings Saved', 'wcag-accessible-pro' ),
            'updated'
        );
    }
    
    settings_errors( 'wcag_accessibility_messages' );
    ?>
    <div class="wrap">
        <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

        <div class="wcag-admin-header" style="background: linear-gradient(135deg, #0066cc 0%, #004499 100%); color: #fff; padding: 30px; border-radius: 8px; margin-bottom: 30px; box-shadow: 0 4px 12px rgba(0,0,0,0.1);">
            <h2 style="margin: 0 0 15px 0; color: #fff; font-size: 28px;">🎯 WCAG 2.1 Level AAA Theme</h2>
            <div style="display: flex; gap: 30px; align-items: center; flex-wrap: wrap;">
                <div style="flex: 1; min-width: 300px;">
                    <p style="margin: 0 0 10px 0; font-size: 16px; opacity: 0.9;">
                        <strong><?php esc_html_e( 'Current Status:', 'wcag-accessible-pro' ); ?></strong>
                        <?php esc_html_e( '100% WCAG 2.1 Level AAA Compliant', 'wcag-accessible-pro' ); ?>
                    </p>
                    <p style="margin: 0; font-size: 14px; opacity: 0.8;">
                        <?php esc_html_e( '78/78 Success Criteria erfüllt', 'wcag-accessible-pro' ); ?>
                    </p>
                </div>
                <div style="background: rgba(255,255,255,0.2); padding: 20px; border-radius: 8px; text-align: center; min-width: 200px;">
                    <div style="font-size: 48px; font-weight: 700; line-height: 1;">100%</div>
                    <div style="font-size: 14px; opacity: 0.9; margin-top: 5px;">Barrierefreiheit</div>
                </div>
            </div>
        </div>

        <div class="notice notice-info">
            <p><strong>💡 <?php esc_html_e( 'Hinweis:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Alle Features sind standardmäßig aktiviert. Deaktivieren Sie Features, die Sie bereits manuell implementiert haben.', 'wcag-accessible-pro' ); ?></p>
        </div>
        
        <form action="options.php" method="post">
            <?php
            settings_fields( 'wcag_accessibility' );
            do_settings_sections( 'wcag_accessibility' );
            submit_button( __( 'Save Settings', 'wcag-accessible-pro' ) );
            ?>
        </form>
        
        <div class="wcag-settings-info" style="margin-top: 40px; padding: 20px; background: #f8f9fa; border-left: 4px solid #0066cc; border-radius: 4px;">
            <h2><?php esc_html_e( 'Feature Overview', 'wcag-accessible-pro' ); ?></h2>
            
            <h3><?php esc_html_e( 'Core Features (Always Active)', 'wcag-accessible-pro' ); ?></h3>
            <ul style="list-style: disc; margin-left: 20px;">
                <li><strong><?php esc_html_e( 'Semantic HTML5:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Native <header>, <nav>, <main>, <article>, <footer> tags', 'wcag-accessible-pro' ); ?></li>
                <li><strong><?php esc_html_e( 'ARIA Landmarks:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'role="banner", role="navigation", role="main", etc.', 'wcag-accessible-pro' ); ?></li>
                <li><strong><?php esc_html_e( 'Focus Management:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( '3px outline with 2px offset', 'wcag-accessible-pro' ); ?></li>
                <li><strong><?php esc_html_e( 'Color Contrast:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( '7:1 ratio (AAA level)', 'wcag-accessible-pro' ); ?></li>
                <li><strong><?php esc_html_e( 'Touch Targets:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Minimum 44x44px', 'wcag-accessible-pro' ); ?></li>
                <li><strong><?php esc_html_e( 'Keyboard Navigation:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Full keyboard accessibility', 'wcag-accessible-pro' ); ?></li>
            </ul>
            
            <h3><?php esc_html_e( 'Optional Features (Can be disabled above)', 'wcag-accessible-pro' ); ?></h3>
            <ul style="list-style: disc; margin-left: 20px;">
                <li><strong><?php esc_html_e( 'Breadcrumb Navigation:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Hierarchical navigation with Schema.org markup', 'wcag-accessible-pro' ); ?></li>
                <li><strong><?php esc_html_e( 'Accessibility Toolbar:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Font size, Dark mode, High contrast controls', 'wcag-accessible-pro' ); ?></li>
                <li><strong><?php esc_html_e( 'Session Timeout Warning:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Warns users 5 minutes before session expires', 'wcag-accessible-pro' ); ?></li>
                <li><strong><?php esc_html_e( 'Autocomplete Attributes:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Automatic autocomplete for form fields', 'wcag-accessible-pro' ); ?></li>
                <li><strong><?php esc_html_e( 'Video Accessibility:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Caption support, admin notices, WebVTT templates', 'wcag-accessible-pro' ); ?></li>
                <li><strong><?php esc_html_e( 'Heading Validation:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Validates H1-H6 hierarchy', 'wcag-accessible-pro' ); ?></li>
                <li><strong><?php esc_html_e( 'Skip Links:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Skip to main content, navigation, footer', 'wcag-accessible-pro' ); ?></li>
                <li><strong><?php esc_html_e( 'Elementor Enhancements:', 'wcag-accessible-pro' ); ?></strong> <?php esc_html_e( 'Semantic wrappers, ARIA, custom controls', 'wcag-accessible-pro' ); ?></li>
            </ul>
            
            <h3><?php esc_html_e( 'Documentation', 'wcag-accessible-pro' ); ?></h3>
            <p>
                <?php
                $docs = array(
                    'README.md'                              => __( 'Theme Documentation', 'wcag-accessible-pro' ),
                    'WCAG-CHECKLIST.md'                      => __( 'WCAG 2.1 Checklist', 'wcag-accessible-pro' ),
                    'GOOGLE-STYLE-GUIDE-COMPLIANCE.md'       => __( 'Google Style Guide Compliance', 'wcag-accessible-pro' ),
                    'NATIVE-SEMANTIC-HTML-INTEGRATION.md'    => __( 'Native Semantic HTML Integration', 'wcag-accessible-pro' ),
                    'COMPLETE-WCAG-FEATURES.md'              => __( 'Complete Feature List', 'wcag-accessible-pro' ),
                );
                
                foreach ( $docs as $file => $label ) {
                    $file_path = get_template_directory() . '/' . $file;
                    if ( file_exists( $file_path ) ) {
                        echo '<a href="' . esc_url( get_template_directory_uri() . '/' . $file ) . '" target="_blank" style="margin-right: 15px;">' . esc_html( $label ) . '</a>';
                    }
                }
                ?>
            </p>
        </div>
    </div>
    
    <style>
    .wcag-settings-info h2 {
        margin-top: 0;
        color: #0066cc;
    }
    
    .wcag-settings-info h3 {
        margin-top: 20px;
        margin-bottom: 10px;
        color: #333;
    }
    
    .wcag-settings-info ul {
        line-height: 1.8;
    }
    
    .wcag-settings-info a {
        color: #0066cc;
        text-decoration: none;
        font-weight: 600;
    }
    
    .wcag-settings-info a:hover {
        text-decoration: underline;
    }
    </style>
    <?php
}

/**
 * Helper function to check if a feature is enabled
 */
function wcag_is_feature_enabled( $feature_name ) {
    $options = get_option( 'wcag_accessibility_options' );
    
    // Default to true if option doesn't exist yet
    if ( ! is_array( $options ) || ! isset( $options[ $feature_name ] ) ) {
        return true;
    }
    
    return (bool) $options[ $feature_name ];
}

