<?php
/**
 * WCAG Accessible Pro Theme Functions
 *
 * IMPORTANT: Native Semantic HTML5 Implementation
 * - All HTML is rendered SERVER-SIDE with proper semantic tags
 * - NO JavaScript manipulation of structure
 * - Optimized for Google Crawler (reads initial HTML, not JS changes)
 * - 100% Google HTML/CSS Style Guide compliant
 * - WCAG 2.1 Level AAA compliant
 *
 * @package WCAG_Accessible_Pro
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Load Elementor compatibility
require get_template_directory() . '/elementor-compatibility.php';

// Load admin settings panel (must be loaded first for wcag_is_feature_enabled function)
require get_template_directory() . '/inc/admin-settings.php';

// Load WCAG enhancements (AAA compliance features)
require get_template_directory() . '/inc/wcag-enhancements.php';

// Load breadcrumb navigation (WCAG 2.4.8)
require get_template_directory() . '/inc/breadcrumbs.php';

// Load video accessibility features (WCAG 1.2.x)
require get_template_directory() . '/inc/video-accessibility.php';

/**
 * Theme Setup
 */
function wcag_accessible_pro_setup() {
    // Make theme available for translation
    load_theme_textdomain( 'wcag-accessible-pro', get_template_directory() . '/languages' );

    // Add default posts and comments RSS feed links to head
    add_theme_support( 'automatic-feed-links' );

    // Let WordPress manage the document title
    add_theme_support( 'title-tag' );

    // Enable support for Post Thumbnails
    add_theme_support( 'post-thumbnails' );

    // Register navigation menus
    register_nav_menus(
        array(
            'primary'   => esc_html__( 'Primary Menu', 'wcag-accessible-pro' ),
            'footer'    => esc_html__( 'Footer Menu', 'wcag-accessible-pro' ),
        )
    );

    // Switch default core markup to output valid HTML5
    add_theme_support(
        'html5',
        array(
            'search-form',
            'comment-form',
            'comment-list',
            'gallery',
            'caption',
            'style',
            'script',
            'navigation-widgets',
        )
    );

    // Add theme support for selective refresh for widgets
    add_theme_support( 'customize-selective-refresh-widgets' );

    // Add support for Block Styles
    add_theme_support( 'wp-block-styles' );

    // Add support for full and wide align images
    add_theme_support( 'align-wide' );

    // Add support for editor styles
    add_theme_support( 'editor-styles' );

    // Add support for responsive embedded content
    add_theme_support( 'responsive-embeds' );

    // Add support for custom line height controls
    add_theme_support( 'custom-line-height' );

    // Add support for experimental link color control
    add_theme_support( 'experimental-link-color' );

    // Add support for custom units
    add_theme_support( 'custom-units' );

    // Add support for custom spacing
    add_theme_support( 'custom-spacing' );

    // Elementor Support
    add_theme_support( 'elementor' );
}
add_action( 'after_setup_theme', 'wcag_accessible_pro_setup' );

/**
 * Set the content width in pixels
 */
function wcag_accessible_pro_content_width() {
    $GLOBALS['content_width'] = apply_filters( 'wcag_accessible_pro_content_width', 1200 );
}
add_action( 'after_setup_theme', 'wcag_accessible_pro_content_width', 0 );

/**
 * Enqueue scripts and styles
 */
function wcag_accessible_pro_scripts() {
    // Main stylesheet
    wp_enqueue_style(
        'wcag-accessible-pro-style',
        get_stylesheet_uri(),
        array(),
        wp_get_theme()->get( 'Version' )
    );

    // Accessibility JavaScript
    wp_enqueue_script(
        'wcag-accessible-pro-accessibility',
        get_template_directory_uri() . '/js/accessibility.js',
        array(),
        wp_get_theme()->get( 'Version' ),
        true
    );

    // Skip link focus fix for IE11
    wp_enqueue_script(
        'wcag-accessible-pro-skip-link-focus-fix',
        get_template_directory_uri() . '/js/skip-link-focus-fix.js',
        array(),
        wp_get_theme()->get( 'Version' ),
        true
    );

    // Comment reply script
    if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
        wp_enqueue_script( 'comment-reply' );
    }
}
add_action( 'wp_enqueue_scripts', 'wcag_accessible_pro_scripts' );

/**
 * Add ARIA landmarks to navigation
 */
function wcag_accessible_pro_nav_menu_args( $args ) {
    $args['container'] = 'nav';
    $args['container_aria_label'] = esc_attr__( 'Primary Navigation', 'wcag-accessible-pro' );
    return $args;
}
add_filter( 'wp_nav_menu_args', 'wcag_accessible_pro_nav_menu_args' );

/**
 * Add skip links for accessibility
 */
function wcag_accessible_pro_skip_links() {
    ?>
    <a class="skip-link screen-reader-text" href="#main"><?php esc_html_e( 'Skip to main content', 'wcag-accessible-pro' ); ?></a>
    <a class="skip-link screen-reader-text" href="#primary-navigation"><?php esc_html_e( 'Skip to navigation', 'wcag-accessible-pro' ); ?></a>
    <a class="skip-link screen-reader-text" href="#footer"><?php esc_html_e( 'Skip to footer', 'wcag-accessible-pro' ); ?></a>
    <?php
}
add_action( 'wp_body_open', 'wcag_accessible_pro_skip_links', 5 );

/**
 * Add language attribute to HTML tag
 */
function wcag_accessible_pro_language_attributes( $output ) {
    $lang = get_bloginfo( 'language' );
    if ( $lang ) {
        $output = 'lang="' . esc_attr( $lang ) . '"';
    }
    return $output;
}
add_filter( 'language_attributes', 'wcag_accessible_pro_language_attributes' );

/**
 * Ensure all images have alt attributes
 */
function wcag_accessible_pro_add_alt_tags( $content ) {
    global $post;
    preg_match_all( '/<img[^>]+>/i', $content, $images );
    
    if ( ! empty( $images[0] ) ) {
        foreach ( $images[0] as $image ) {
            if ( strpos( $image, 'alt=' ) === false ) {
                $new_image = str_replace( '<img', '<img alt="' . esc_attr( get_the_title( $post->ID ) ) . '"', $image );
                $content = str_replace( $image, $new_image, $content );
            }
        }
    }
    
    return $content;
}
add_filter( 'the_content', 'wcag_accessible_pro_add_alt_tags' );

/**
 * Add ARIA labels to read more links
 */
function wcag_accessible_pro_excerpt_more( $more ) {
    if ( ! is_single() ) {
        global $post;
        $more = sprintf(
            '... <a href="%1$s" class="more-link" aria-label="%2$s">%3$s</a>',
            esc_url( get_permalink( $post->ID ) ),
            /* translators: %s: Post title */
            esc_attr( sprintf( __( 'Continue reading %s', 'wcag-accessible-pro' ), get_the_title( $post->ID ) ) ),
            esc_html__( 'Read more', 'wcag-accessible-pro' )
        );
    }
    return $more;
}
add_filter( 'excerpt_more', 'wcag_accessible_pro_excerpt_more' );

/**
 * Add proper heading hierarchy to archive titles
 */
function wcag_accessible_pro_archive_title( $title ) {
    if ( is_category() ) {
        $title = single_cat_title( '', false );
    } elseif ( is_tag() ) {
        $title = single_tag_title( '', false );
    } elseif ( is_author() ) {
        $title = get_the_author();
    } elseif ( is_post_type_archive() ) {
        $title = post_type_archive_title( '', false );
    } elseif ( is_tax() ) {
        $title = single_term_title( '', false );
    }
    
    return $title;
}
add_filter( 'get_the_archive_title', 'wcag_accessible_pro_archive_title' );

/**
 * Add ARIA labels to pagination
 */
function wcag_accessible_pro_pagination_args( $args ) {
    $args['aria_label'] = esc_attr__( 'Posts pagination', 'wcag-accessible-pro' );
    $args['prev_text'] = '<span aria-hidden="true">&laquo;</span> <span class="screen-reader-text">' . esc_html__( 'Previous page', 'wcag-accessible-pro' ) . '</span>';
    $args['next_text'] = '<span class="screen-reader-text">' . esc_html__( 'Next page', 'wcag-accessible-pro' ) . '</span> <span aria-hidden="true">&raquo;</span>';
    return $args;
}
add_filter( 'paginate_links_args', 'wcag_accessible_pro_pagination_args' );

/**
 * Add proper ARIA attributes to comment form
 */
function wcag_accessible_pro_comment_form_defaults( $defaults ) {
    $defaults['title_reply_before'] = '<h3 id="reply-title" class="comment-reply-title">';
    $defaults['title_reply_after'] = '</h3>';
    $defaults['comment_field'] = '<p class="comment-form-comment"><label for="comment">' . esc_html__( 'Comment', 'wcag-accessible-pro' ) . ' <span class="required" aria-label="' . esc_attr__( 'Required', 'wcag-accessible-pro' ) . '">*</span></label><textarea id="comment" name="comment" cols="45" rows="8" maxlength="65525" required="required" aria-required="true"></textarea></p>';
    
    return $defaults;
}
add_filter( 'comment_form_defaults', 'wcag_accessible_pro_comment_form_defaults' );

/**
 * Elementor Compatibility
 */
function wcag_accessible_pro_register_elementor_locations( $elementor_theme_manager ) {
    $elementor_theme_manager->register_all_core_location();
}
add_action( 'elementor/theme/register_locations', 'wcag_accessible_pro_register_elementor_locations' );

/**
 * Add Elementor support for custom breakpoints
 */
function wcag_accessible_pro_elementor_breakpoints() {
    return array(
        'xs' => 0,
        'sm' => 576,
        'md' => 768,
        'lg' => 1024,
        'xl' => 1200,
        'xxl' => 1400,
    );
}
add_filter( 'elementor/breakpoints/get_breakpoints', 'wcag_accessible_pro_elementor_breakpoints' );

/**
 * Register widget areas
 */
function wcag_accessible_pro_widgets_init() {
    register_sidebar(
        array(
            'name'          => esc_html__( 'Sidebar', 'wcag-accessible-pro' ),
            'id'            => 'sidebar-1',
            'description'   => esc_html__( 'Add widgets here.', 'wcag-accessible-pro' ),
            'before_widget' => '<section id="%1$s" class="widget %2$s">',
            'after_widget'  => '</section>',
            'before_title'  => '<h2 class="widget-title">',
            'after_title'   => '</h2>',
        )
    );
}
add_action( 'widgets_init', 'wcag_accessible_pro_widgets_init' );

/**
 * Add Schema.org markup for better SEO
 */
function wcag_accessible_pro_schema_markup() {
    $schema = array(
        '@context' => 'https://schema.org',
        '@type' => 'WebSite',
        'name' => get_bloginfo( 'name' ),
        'url' => home_url( '/' ),
        'description' => get_bloginfo( 'description' ),
    );
    
    if ( is_singular() ) {
        global $post;
        $schema = array(
            '@context' => 'https://schema.org',
            '@type' => 'Article',
            'headline' => get_the_title(),
            'datePublished' => get_the_date( 'c' ),
            'dateModified' => get_the_modified_date( 'c' ),
            'author' => array(
                '@type' => 'Person',
                'name' => get_the_author(),
            ),
        );
    }
    
    echo '<script type="application/ld+json">' . wp_json_encode( $schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE ) . '</script>';
}
add_action( 'wp_head', 'wcag_accessible_pro_schema_markup' );

