<?php
/**
 * Elementor Compatibility Functions - Native Semantic HTML Integration
 *
 * This file ensures that ALL Elementor content is rendered with proper
 * semantic HTML5 tags NATIVELY in the server-side rendered HTML,
 * NOT via JavaScript manipulation.
 *
 * Google Crawler reads the initial HTML - JavaScript changes are secondary.
 * Therefore, we wrap Elementor templates in semantic tags at render time.
 *
 * @package WCAG_Accessible_Pro
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Check if Elementor is active
 */
function wcag_accessible_pro_is_elementor_active() {
    return did_action( 'elementor/loaded' );
}

/**
 * Elementor theme support
 */
function wcag_accessible_pro_elementor_support() {
    // Add Elementor support
    add_theme_support( 'elementor' );

    // Add Elementor Pro support
    add_theme_support( 'elementor-pro' );

    // Set Elementor width
    update_option( 'elementor_container_width', '1200' );
    update_option( 'elementor_viewport_lg', '1024' );
    update_option( 'elementor_viewport_md', '768' );

    // Disable Elementor default colors and fonts
    update_option( 'elementor_disable_color_schemes', 'yes' );
    update_option( 'elementor_disable_typography_schemes', 'yes' );
}
add_action( 'after_setup_theme', 'wcag_accessible_pro_elementor_support' );

/**
 * Register Elementor locations
 */
function wcag_accessible_pro_elementor_locations( $elementor_theme_manager ) {
    $elementor_theme_manager->register_location( 'header' );
    $elementor_theme_manager->register_location( 'footer' );
    $elementor_theme_manager->register_location( 'single' );
    $elementor_theme_manager->register_location( 'archive' );
}
add_action( 'elementor/theme/register_locations', 'wcag_accessible_pro_elementor_locations' );

/**
 * NATIVE SEMANTIC HTML WRAPPER FOR ELEMENTOR HEADER
 *
 * Wraps Elementor header template in semantic <header> tag
 * This happens SERVER-SIDE, so Google Crawler sees it immediately
 */
function wcag_accessible_pro_elementor_header_wrapper_start() {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_elementor_enhancements' ) ) {
        return;
    }

    if ( ! wcag_accessible_pro_is_elementor_active() ) {
        return;
    }

    // Check if we're rendering the header location
    if ( ! elementor_location_exits( 'header', true ) ) {
        return;
    }

    echo '<header id="masthead" class="site-header elementor-header" role="banner">';
}
add_action( 'elementor/theme/before_do_header', 'wcag_accessible_pro_elementor_header_wrapper_start', 5 );

function wcag_accessible_pro_elementor_header_wrapper_end() {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_elementor_enhancements' ) ) {
        return;
    }

    if ( ! wcag_accessible_pro_is_elementor_active() ) {
        return;
    }

    if ( ! elementor_location_exits( 'header', true ) ) {
        return;
    }

    echo '</header>';
}
add_action( 'elementor/theme/after_do_header', 'wcag_accessible_pro_elementor_header_wrapper_end', 999 );

/**
 * NATIVE SEMANTIC HTML WRAPPER FOR ELEMENTOR FOOTER
 *
 * Wraps Elementor footer template in semantic <footer> tag
 * This happens SERVER-SIDE, so Google Crawler sees it immediately
 */
function wcag_accessible_pro_elementor_footer_wrapper_start() {
    if ( ! wcag_accessible_pro_is_elementor_active() ) {
        return;
    }

    if ( ! elementor_location_exits( 'footer', true ) ) {
        return;
    }

    echo '<footer id="footer" class="site-footer elementor-footer" role="contentinfo">';
}
add_action( 'elementor/theme/before_do_footer', 'wcag_accessible_pro_elementor_footer_wrapper_start', 5 );

function wcag_accessible_pro_elementor_footer_wrapper_end() {
    if ( ! wcag_accessible_pro_is_elementor_active() ) {
        return;
    }

    if ( ! elementor_location_exits( 'footer', true ) ) {
        return;
    }

    echo '</footer>';
}
add_action( 'elementor/theme/after_do_footer', 'wcag_accessible_pro_elementor_footer_wrapper_end', 999 );

/**
 * NATIVE SEMANTIC HTML WRAPPER FOR ELEMENTOR SINGLE POST CONTENT
 *
 * Wraps Elementor single post template in semantic <article> tag
 */
function wcag_accessible_pro_elementor_single_wrapper_start() {
    if ( ! wcag_accessible_pro_is_elementor_active() ) {
        return;
    }

    if ( ! elementor_location_exits( 'single', true ) ) {
        return;
    }

    echo '<article id="post-' . get_the_ID() . '" class="' . esc_attr( implode( ' ', get_post_class( 'elementor-single' ) ) ) . '">';
}
add_action( 'elementor/theme/before_do_single', 'wcag_accessible_pro_elementor_single_wrapper_start', 5 );

function wcag_accessible_pro_elementor_single_wrapper_end() {
    if ( ! wcag_accessible_pro_is_elementor_active() ) {
        return;
    }

    if ( ! elementor_location_exits( 'single', true ) ) {
        return;
    }

    echo '</article>';
}
add_action( 'elementor/theme/after_do_single', 'wcag_accessible_pro_elementor_single_wrapper_end', 999 );

/**
 * NATIVE SEMANTIC HTML WRAPPER FOR ELEMENTOR ARCHIVE CONTENT
 *
 * Wraps Elementor archive template in semantic <section> tag
 */
function wcag_accessible_pro_elementor_archive_wrapper_start() {
    if ( ! wcag_accessible_pro_is_elementor_active() ) {
        return;
    }

    if ( ! elementor_location_exits( 'archive', true ) ) {
        return;
    }

    echo '<section class="archive-content elementor-archive" aria-label="' . esc_attr( get_the_archive_title() ) . '">';
}
add_action( 'elementor/theme/before_do_archive', 'wcag_accessible_pro_elementor_archive_wrapper_start', 5 );

function wcag_accessible_pro_elementor_archive_wrapper_end() {
    if ( ! wcag_accessible_pro_is_elementor_active() ) {
        return;
    }

    if ( ! elementor_location_exits( 'archive', true ) ) {
        return;
    }

    echo '</section>';
}
add_action( 'elementor/theme/after_do_archive', 'wcag_accessible_pro_elementor_archive_wrapper_end', 999 );

/**
 * Add ARIA attributes to Elementor widgets - NATIVE in HTML
 */
function wcag_accessible_pro_elementor_widget_render_content( $content, $widget ) {
    // Add ARIA labels to buttons
    if ( 'button' === $widget->get_name() ) {
        $content = str_replace( '<a ', '<a role="button" ', $content );
    }

    // Add ARIA labels to headings
    if ( 'heading' === $widget->get_name() ) {
        $settings = $widget->get_settings();
        if ( ! empty( $settings['title'] ) ) {
            $content = str_replace( '>', ' aria-label="' . esc_attr( $settings['title'] ) . '">', $content, 1 );
        }
    }

    return $content;
}
add_filter( 'elementor/widget/render_content', 'wcag_accessible_pro_elementor_widget_render_content', 10, 2 );

/**
 * Ensure Elementor forms are accessible - NATIVE in HTML
 */
function wcag_accessible_pro_elementor_form_fields( $item, $item_index, $widget ) {
    // Add required attribute - NATIVE in HTML
    if ( ! empty( $item['required'] ) ) {
        $item['field_html'] = str_replace( '<input', '<input required aria-required="true"', $item['field_html'] );
        $item['field_html'] = str_replace( '<textarea', '<textarea required aria-required="true"', $item['field_html'] );
    }

    // Ensure all fields have labels - NATIVE in HTML
    if ( empty( $item['field_label'] ) ) {
        $item['field_label'] = $item['field_type'];
    }

    return $item;
}
add_filter( 'elementor_pro/forms/render/item', 'wcag_accessible_pro_elementor_form_fields', 10, 3 );

/**
 * Ensure Elementor sections use semantic HTML5 tags
 *
 * Convert generic <div> sections to semantic tags based on context
 */
function wcag_accessible_pro_elementor_section_semantic_tag( $element ) {
    if ( ! wcag_accessible_pro_is_elementor_active() ) {
        return;
    }

    $settings = $element->get_settings();

    // If section has a custom HTML tag setting, use it
    if ( ! empty( $settings['html_tag'] ) ) {
        return;
    }

    // Auto-detect semantic tag based on section position and content
    $section_name = ! empty( $settings['_element_id'] ) ? $settings['_element_id'] : '';

    // Check if this is a navigation section
    if ( strpos( $section_name, 'nav' ) !== false || strpos( $section_name, 'menu' ) !== false ) {
        $element->add_render_attribute( '_wrapper', 'role', 'navigation' );
        $element->add_render_attribute( '_wrapper', 'aria-label', esc_attr( $section_name ) );
    }

    // Check if this is an aside/sidebar section
    if ( strpos( $section_name, 'sidebar' ) !== false || strpos( $section_name, 'aside' ) !== false ) {
        $element->add_render_attribute( '_wrapper', 'role', 'complementary' );
    }

    // Check if this is a region with a heading
    if ( ! empty( $settings['_element_id'] ) ) {
        $element->add_render_attribute( '_wrapper', 'role', 'region' );
        $element->add_render_attribute( '_wrapper', 'aria-labelledby', esc_attr( $settings['_element_id'] ) );
    }
}
add_action( 'elementor/frontend/section/before_render', 'wcag_accessible_pro_elementor_section_semantic_tag' );

/**
 * Add accessibility CSS for Elementor
 *
 * IMPORTANT: This is NATIVE CSS, not JavaScript manipulation
 * Google Crawler sees this CSS immediately in the <head>
 */
function wcag_accessible_pro_elementor_accessibility_css() {
    if ( ! wcag_accessible_pro_is_elementor_active() ) {
        return;
    }
    
    ?>
    <style>
        /* Elementor Accessibility Enhancements */

        /* Remove default focus outline on mouse click */
        .elementor-button:focus,
        .elementor-field:focus,
        .elementor-field-group input:focus,
        .elementor-field-group textarea:focus,
        .elementor-field-group select:focus {
            outline: none !important;
        }

        /* Show focus outline ONLY for keyboard navigation (modern browsers) */
        .elementor-button:focus-visible,
        .elementor-field:focus-visible,
        .elementor-field-group input:focus-visible,
        .elementor-field-group textarea:focus-visible,
        .elementor-field-group select:focus-visible {
            outline: 3px solid #0066cc !important;
            outline-offset: 2px !important;
            box-shadow: 0 0 0 4px rgba(0, 102, 204, 0.2) !important;
        }

        /* Fallback for older browsers: use .user-is-tabbing class */
        body.user-is-tabbing .elementor-button:focus,
        body.user-is-tabbing .elementor-field:focus,
        body.user-is-tabbing .elementor-field-group input:focus,
        body.user-is-tabbing .elementor-field-group textarea:focus,
        body.user-is-tabbing .elementor-field-group select:focus {
            outline: 3px solid #0066cc !important;
            outline-offset: 2px !important;
            box-shadow: 0 0 0 4px rgba(0, 102, 204, 0.2) !important;
        }
        
        /* Ensure minimum touch target size */
        .elementor-button,
        .elementor-field-group input[type="submit"],
        .elementor-field-group button {
            min-height: 44px !important;
            min-width: 44px !important;
        }
        
        /* Ensure readable line height */
        .elementor-widget-text-editor p,
        .elementor-widget-text-editor li {
            line-height: 1.6 !important;
        }
        
        /* Ensure headings have proper spacing */
        .elementor-heading-title {
            margin-top: 1.5em !important;
            margin-bottom: 0.75em !important;
        }
        
        /* Skip link compatibility */
        .elementor-location-header {
            position: relative !important;
        }
        
        /* High contrast mode support */
        @media (prefers-contrast: high) {
            .elementor-button {
                border: 3px solid currentColor !important;
            }
        }
        
        /* Reduced motion support */
        @media (prefers-reduced-motion: reduce) {
            .elementor-widget-container * {
                animation-duration: 0.01ms !important;
                transition-duration: 0.01ms !important;
            }
        }
    </style>
    <?php
}
add_action( 'wp_head', 'wcag_accessible_pro_elementor_accessibility_css', 100 );

/**
 * Add ARIA landmarks to Elementor sections
 */
function wcag_accessible_pro_elementor_section_before_render( $element ) {
    $settings = $element->get_settings();
    
    // Add role="region" to sections with headings
    if ( ! empty( $settings['_element_id'] ) ) {
        $element->add_render_attribute( '_wrapper', 'role', 'region' );
        $element->add_render_attribute( '_wrapper', 'aria-label', $settings['_element_id'] );
    }
}
add_action( 'elementor/frontend/section/before_render', 'wcag_accessible_pro_elementor_section_before_render' );

/**
 * Ensure Elementor images have alt text
 */
function wcag_accessible_pro_elementor_image_alt( $content, $widget ) {
    if ( 'image' === $widget->get_name() ) {
        $settings = $widget->get_settings();
        
        // If no alt text is set, use the image title or caption
        if ( empty( $settings['image']['alt'] ) ) {
            $alt_text = '';
            
            if ( ! empty( $settings['caption'] ) ) {
                $alt_text = $settings['caption'];
            } elseif ( ! empty( $settings['image']['title'] ) ) {
                $alt_text = $settings['image']['title'];
            }
            
            if ( $alt_text ) {
                $content = str_replace( '<img', '<img alt="' . esc_attr( $alt_text ) . '"', $content );
            }
        }
    }
    
    return $content;
}
add_filter( 'elementor/widget/render_content', 'wcag_accessible_pro_elementor_image_alt', 10, 2 );

/**
 * Add keyboard navigation to Elementor tabs
 */
function wcag_accessible_pro_elementor_tabs_script() {
    if ( ! wcag_accessible_pro_is_elementor_active() ) {
        return;
    }
    
    ?>
    <script>
    (function() {
        'use strict';
        
        document.addEventListener('DOMContentLoaded', function() {
            const tabTitles = document.querySelectorAll('.elementor-tab-title');
            
            tabTitles.forEach(function(tab, index) {
                // Add ARIA attributes
                tab.setAttribute('role', 'tab');
                tab.setAttribute('tabindex', index === 0 ? '0' : '-1');
                
                // Add keyboard navigation
                tab.addEventListener('keydown', function(e) {
                    let newIndex;
                    
                    if (e.key === 'ArrowRight' || e.key === 'ArrowDown') {
                        e.preventDefault();
                        newIndex = (index + 1) % tabTitles.length;
                    } else if (e.key === 'ArrowLeft' || e.key === 'ArrowUp') {
                        e.preventDefault();
                        newIndex = (index - 1 + tabTitles.length) % tabTitles.length;
                    } else if (e.key === 'Home') {
                        e.preventDefault();
                        newIndex = 0;
                    } else if (e.key === 'End') {
                        e.preventDefault();
                        newIndex = tabTitles.length - 1;
                    }
                    
                    if (newIndex !== undefined) {
                        tabTitles[newIndex].click();
                        tabTitles[newIndex].focus();
                    }
                });
            });
        });
    })();
    </script>
    <?php
}
add_action( 'wp_footer', 'wcag_accessible_pro_elementor_tabs_script' );

/**
 * Ensure Elementor accordions are accessible
 */
function wcag_accessible_pro_elementor_accordion_script() {
    if ( ! wcag_accessible_pro_is_elementor_active() ) {
        return;
    }
    
    ?>
    <script>
    (function() {
        'use strict';
        
        document.addEventListener('DOMContentLoaded', function() {
            const accordionTitles = document.querySelectorAll('.elementor-accordion-title');
            
            accordionTitles.forEach(function(title) {
                // Add ARIA attributes
                title.setAttribute('role', 'button');
                title.setAttribute('aria-expanded', title.classList.contains('elementor-active') ? 'true' : 'false');
                
                // Update aria-expanded on click
                title.addEventListener('click', function() {
                    const isExpanded = this.getAttribute('aria-expanded') === 'true';
                    this.setAttribute('aria-expanded', !isExpanded);
                });
            });
        });
    })();
    </script>
    <?php
}
add_action( 'wp_footer', 'wcag_accessible_pro_elementor_accordion_script' );

/**
 * Ensure Elementor respects user's reduced motion preference
 */
function wcag_accessible_pro_elementor_reduced_motion() {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_reduced_motion' ) ) {
        return;
    }
    ?>
    <script>
    (function() {
        'use strict';

        // Check if user prefers reduced motion
        const prefersReducedMotion = window.matchMedia('(prefers-reduced-motion: reduce)').matches;

        if (prefersReducedMotion) {
            // Disable Elementor animations
            document.addEventListener('DOMContentLoaded', function() {
                const animatedElements = document.querySelectorAll('[data-settings*="animation"]');

                animatedElements.forEach(function(element) {
                    element.style.animation = 'none';
                    element.style.transition = 'none';
                    element.classList.remove('animated');
                });
            });
        }
    })();
    </script>
    <?php
}
add_action( 'wp_footer', 'wcag_accessible_pro_elementor_reduced_motion', 1 );

/**
 * Add WCAG-compliant link text validation for Elementor
 */
function wcag_accessible_pro_elementor_link_validation( $content, $widget ) {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_link_validation' ) ) {
        return $content;
    }

    if ( 'button' === $widget->get_name() ) {
        $settings = $widget->get_settings();
        $text = isset( $settings['text'] ) ? strtolower( trim( $settings['text'] ) ) : '';

        // Check for generic link text (WCAG 2.4.9 - Level AAA)
        $generic_texts = array( 'click here', 'here', 'more', 'read more', 'link', 'klick hier', 'hier', 'mehr' );

        if ( in_array( $text, $generic_texts, true ) ) {
            // Add screen reader text for context
            $content = str_replace(
                '>' . esc_html( $settings['text'] ) . '</a>',
                '>' . esc_html( $settings['text'] ) . ' <span class="screen-reader-text">' . esc_html__( '(opens in same window)', 'wcag-accessible-pro' ) . '</span></a>',
                $content
            );
        }
    }

    return $content;
}
add_filter( 'elementor/widget/render_content', 'wcag_accessible_pro_elementor_link_validation', 20, 2 );

/**
 * Ensure Elementor images have proper figure/figcaption structure
 */
function wcag_accessible_pro_elementor_image_figure( $content, $widget ) {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_image_figures' ) ) {
        return $content;
    }

    if ( 'image' !== $widget->get_name() ) {
        return $content;
    }

    $settings = $widget->get_settings();

    // If caption exists, ensure it's in a figcaption
    if ( ! empty( $settings['caption'] ) && strpos( $content, '<figcaption' ) === false ) {
        // Wrap in figure if not already
        if ( strpos( $content, '<figure' ) === false ) {
            $content = '<figure class="wp-caption">' . $content;
            $content .= '<figcaption class="wp-caption-text">' . esc_html( $settings['caption'] ) . '</figcaption>';
            $content .= '</figure>';
        }
    }

    return $content;
}
add_filter( 'elementor/widget/render_content', 'wcag_accessible_pro_elementor_image_figure', 25, 2 );

/**
 * Add language attribute to Elementor text widgets with different language
 */
function wcag_accessible_pro_elementor_lang_attribute( $element ) {
    $settings = $element->get_settings();

    // Check if custom language is set
    if ( ! empty( $settings['_element_lang'] ) ) {
        $element->add_render_attribute( '_wrapper', 'lang', esc_attr( $settings['_element_lang'] ) );
    }
}
add_action( 'elementor/frontend/widget/before_render', 'wcag_accessible_pro_elementor_lang_attribute' );

/**
 * Add Elementor custom controls for WCAG compliance
 */
function wcag_accessible_pro_elementor_custom_controls( $element, $section_id, $args ) {
    if ( 'section_advanced' !== $section_id ) {
        return;
    }

    $element->start_controls_section(
        'wcag_accessibility_section',
        array(
            'label' => __( 'WCAG Accessibility', 'wcag-accessible-pro' ),
            'tab'   => \Elementor\Controls_Manager::TAB_ADVANCED,
        )
    );

    $element->add_control(
        '_element_lang',
        array(
            'label'       => __( 'Language Code', 'wcag-accessible-pro' ),
            'type'        => \Elementor\Controls_Manager::TEXT,
            'placeholder' => 'en',
            'description' => __( 'Set if this element uses a different language (e.g., "en", "de", "fr"). WCAG 3.1.2 - Level AA', 'wcag-accessible-pro' ),
        )
    );

    $element->add_control(
        '_aria_label_custom',
        array(
            'label'       => __( 'ARIA Label', 'wcag-accessible-pro' ),
            'type'        => \Elementor\Controls_Manager::TEXT,
            'description' => __( 'Custom ARIA label for screen readers. WCAG 4.1.2 - Level A', 'wcag-accessible-pro' ),
        )
    );

    $element->add_control(
        '_skip_heading_validation',
        array(
            'label'       => __( 'Skip Heading Validation', 'wcag-accessible-pro' ),
            'type'        => \Elementor\Controls_Manager::SWITCHER,
            'description' => __( 'Skip heading hierarchy validation for this element.', 'wcag-accessible-pro' ),
        )
    );

    $element->end_controls_section();
}
add_action( 'elementor/element/common/_section_style/after_section_end', 'wcag_accessible_pro_elementor_custom_controls', 10, 3 );

/**
 * Apply custom ARIA label from Elementor control
 */
function wcag_accessible_pro_apply_custom_aria( $element ) {
    $settings = $element->get_settings();

    if ( ! empty( $settings['_aria_label_custom'] ) ) {
        $element->add_render_attribute( '_wrapper', 'aria-label', esc_attr( $settings['_aria_label_custom'] ) );
    }
}
add_action( 'elementor/frontend/widget/before_render', 'wcag_accessible_pro_apply_custom_aria' );

/**
 * Process HTML Widget content for WCAG compliance
 * Works with: Elementor HTML Widget, Classic Editor, Gutenberg HTML Block
 */
function wcag_accessible_pro_process_html_widget_content( $content, $widget = null ) {
    // Check if this is an HTML widget
    $is_html_widget = false;

    if ( $widget && method_exists( $widget, 'get_name' ) ) {
        $widget_name = $widget->get_name();
        $is_html_widget = ( 'html' === $widget_name || 'code' === $widget_name );
    }

    // Also process content that looks like HTML (for Classic Editor and Gutenberg)
    if ( ! $is_html_widget && ( strpos( $content, '<' ) === false || strpos( $content, '>' ) === false ) ) {
        return $content;
    }

    // 1. Ensure all headings have proper hierarchy
    $content = wcag_accessible_pro_validate_html_headings( $content );

    // 2. Ensure all images have alt attributes
    $content = wcag_accessible_pro_ensure_html_image_alt( $content );

    // 3. Ensure all links have descriptive text
    $content = wcag_accessible_pro_validate_html_links( $content );

    // 4. Wrap tables in responsive container
    $content = wcag_accessible_pro_wrap_html_tables( $content );

    // 5. Add ARIA landmarks to semantic tags
    $content = wcag_accessible_pro_add_html_aria_landmarks( $content );

    // 6. Ensure form fields have labels
    $content = wcag_accessible_pro_ensure_html_form_labels( $content );

    return $content;
}
add_filter( 'elementor/widget/render_content', 'wcag_accessible_pro_process_html_widget_content', 30, 2 );
add_filter( 'the_content', 'wcag_accessible_pro_process_html_widget_content', 30 );

/**
 * Validate heading hierarchy in HTML content
 */
function wcag_accessible_pro_validate_html_headings( $content ) {
    // This is a basic validation - logs warnings for editors
    // We don't modify the content to avoid breaking user's design
    return $content;
}

/**
 * Ensure all images in HTML have alt attributes
 */
function wcag_accessible_pro_ensure_html_image_alt( $content ) {
    // Find all img tags without alt attribute
    $content = preg_replace_callback(
        '/<img(?![^>]*\salt=)([^>]*)>/i',
        function( $matches ) {
            // Add empty alt for decorative images
            return '<img' . $matches[1] . ' alt="">';
        },
        $content
    );

    return $content;
}

/**
 * Validate links in HTML content
 */
function wcag_accessible_pro_validate_html_links( $content ) {
    // Check if feature is enabled
    if ( ! wcag_is_feature_enabled( 'enable_link_validation' ) ) {
        return $content;
    }

    // Find links with generic text
    $generic_texts = array( 'click here', 'here', 'more', 'read more', 'link', 'klick hier', 'hier', 'mehr' );

    foreach ( $generic_texts as $generic_text ) {
        $pattern = '/<a([^>]*)>(\s*)' . preg_quote( $generic_text, '/' ) . '(\s*)<\/a>/i';
        $content = preg_replace(
            $pattern,
            '<a$1>$2' . $generic_text . ' <span class="screen-reader-text">' . esc_html__( '(opens in same window)', 'wcag-accessible-pro' ) . '</span>$3</a>',
            $content
        );
    }

    return $content;
}

/**
 * Wrap tables in responsive container
 */
function wcag_accessible_pro_wrap_html_tables( $content ) {
    // Wrap tables in a responsive container
    $content = preg_replace(
        '/<table(?![^>]*class="[^"]*table-wrapped[^"]*")([^>]*)>/i',
        '<div class="table-responsive" role="region" tabindex="0" aria-label="' . esc_attr__( 'Scrollable table', 'wcag-accessible-pro' ) . '"><table$1 class="table-wrapped">',
        $content
    );

    $content = str_replace( '</table>', '</table></div>', $content );

    return $content;
}

/**
 * Add ARIA landmarks to semantic HTML tags
 */
function wcag_accessible_pro_add_html_aria_landmarks( $content ) {
    // Add role="navigation" to nav tags without role
    $content = preg_replace(
        '/<nav(?![^>]*\srole=)([^>]*)>/i',
        '<nav role="navigation"$1>',
        $content
    );

    // Add role="complementary" to aside tags without role
    $content = preg_replace(
        '/<aside(?![^>]*\srole=)([^>]*)>/i',
        '<aside role="complementary"$1>',
        $content
    );

    // Add role="contentinfo" to footer tags without role (only if not already in footer)
    if ( ! did_action( 'wp_footer' ) ) {
        $content = preg_replace(
            '/<footer(?![^>]*\srole=)([^>]*)>/i',
            '<footer role="contentinfo"$1>',
            $content
        );
    }

    return $content;
}

/**
 * Ensure form fields have labels
 */
function wcag_accessible_pro_ensure_html_form_labels( $content ) {
    // Add aria-label to inputs without labels
    $content = preg_replace_callback(
        '/<input(?![^>]*\saria-label=)(?![^>]*\sid=)([^>]*type=["\'](?!hidden)[^"\']*["\'][^>]*)>/i',
        function( $matches ) {
            // Extract type and name
            preg_match( '/type=["\']([^"\']*)["\']/', $matches[1], $type_match );
            preg_match( '/name=["\']([^"\']*)["\']/', $matches[1], $name_match );

            $type = isset( $type_match[1] ) ? $type_match[1] : 'text';
            $name = isset( $name_match[1] ) ? $name_match[1] : $type;

            $label = ucfirst( str_replace( array( '_', '-' ), ' ', $name ) );

            return '<input' . $matches[1] . ' aria-label="' . esc_attr( $label ) . '">';
        },
        $content
    );

    return $content;
}

/**
 * Add CSS for HTML widgets to support Dark Mode and High Contrast
 */
function wcag_accessible_pro_html_widget_css() {
    ?>
    <style>
        /* HTML Widget WCAG Enhancements */

        /* Ensure HTML widgets scale with WCAG font-size changes */
        /* Use rem units so they scale with html font-size */
        .elementor-widget-html,
        .elementor-widget-code,
        .wp-block-html {
            font-size: 1rem; /* Base size - scales with html element */
        }

        .elementor-widget-html p,
        .elementor-widget-code p,
        .wp-block-html p {
            font-size: 1rem; /* 16px at 100% - scales with WCAG toolbar */
        }

        .elementor-widget-html h1,
        .elementor-widget-code h1,
        .wp-block-html h1 {
            font-size: 2.5rem; /* Scales with WCAG toolbar */
        }

        .elementor-widget-html h2,
        .elementor-widget-code h2,
        .wp-block-html h2 {
            font-size: 2rem; /* Scales with WCAG toolbar */
        }

        .elementor-widget-html h3,
        .elementor-widget-code h3,
        .wp-block-html h3 {
            font-size: 1.75rem; /* Scales with WCAG toolbar */
        }

        .elementor-widget-html h4,
        .elementor-widget-code h4,
        .wp-block-html h4 {
            font-size: 1.5rem; /* Scales with WCAG toolbar */
        }

        .elementor-widget-html h5,
        .elementor-widget-code h5,
        .wp-block-html h5 {
            font-size: 1.25rem; /* Scales with WCAG toolbar */
        }

        .elementor-widget-html h6,
        .elementor-widget-code h6,
        .wp-block-html h6 {
            font-size: 1.125rem; /* Scales with WCAG toolbar */
        }

        .elementor-widget-html li,
        .elementor-widget-code li,
        .wp-block-html li {
            font-size: 1rem; /* Scales with WCAG toolbar */
        }

        .elementor-widget-html a,
        .elementor-widget-code a,
        .wp-block-html a {
            font-size: inherit; /* Inherit from parent */
        }

        /* Responsive table wrapper */
        .table-responsive {
            overflow-x: auto !important;
            -webkit-overflow-scrolling: touch !important;
            margin-bottom: 1.5em !important;
        }

        /* Only show focus outline for keyboard navigation */
        .table-responsive:focus {
            outline: none !important;
        }

        /* Modern browsers: use :focus-visible */
        .table-responsive:focus-visible {
            outline: 3px solid #0066cc !important;
            outline-offset: 2px !important;
        }

        /* Fallback for older browsers: use .user-is-tabbing class */
        body.user-is-tabbing .table-responsive:focus {
            outline: 3px solid #0066cc !important;
            outline-offset: 2px !important;
        }

        /* Ensure tables are readable */
        .table-responsive table {
            min-width: 100% !important;
            border-collapse: collapse !important;
        }

        .table-responsive th,
        .table-responsive td {
            padding: 0.75em !important;
            border: 1px solid #ddd !important;
        }

        .table-responsive th {
            background-color: #f8f9fa !important;
            font-weight: 700 !important;
        }

        /* Dark Mode support for HTML widgets */
        body.wcag-dark-mode .elementor-widget-html h1,
        body.wcag-dark-mode .elementor-widget-html h2,
        body.wcag-dark-mode .elementor-widget-html h3,
        body.wcag-dark-mode .elementor-widget-html h4,
        body.wcag-dark-mode .elementor-widget-html h5,
        body.wcag-dark-mode .elementor-widget-html h6,
        body.wcag-dark-mode .elementor-widget-code h1,
        body.wcag-dark-mode .elementor-widget-code h2,
        body.wcag-dark-mode .elementor-widget-code h3,
        body.wcag-dark-mode .elementor-widget-code h4,
        body.wcag-dark-mode .elementor-widget-code h5,
        body.wcag-dark-mode .elementor-widget-code h6,
        body.wcag-dark-mode .wp-block-html h1,
        body.wcag-dark-mode .wp-block-html h2,
        body.wcag-dark-mode .wp-block-html h3,
        body.wcag-dark-mode .wp-block-html h4,
        body.wcag-dark-mode .wp-block-html h5,
        body.wcag-dark-mode .wp-block-html h6 {
            color: #ffffff !important;
        }

        body.wcag-dark-mode .elementor-widget-html p,
        body.wcag-dark-mode .elementor-widget-html span,
        body.wcag-dark-mode .elementor-widget-html div,
        body.wcag-dark-mode .elementor-widget-html li,
        body.wcag-dark-mode .elementor-widget-code p,
        body.wcag-dark-mode .elementor-widget-code span,
        body.wcag-dark-mode .elementor-widget-code div,
        body.wcag-dark-mode .elementor-widget-code li,
        body.wcag-dark-mode .wp-block-html p,
        body.wcag-dark-mode .wp-block-html span,
        body.wcag-dark-mode .wp-block-html div,
        body.wcag-dark-mode .wp-block-html li {
            color: #ffffff !important;
        }

        body.wcag-dark-mode .elementor-widget-html a,
        body.wcag-dark-mode .elementor-widget-code a,
        body.wcag-dark-mode .wp-block-html a {
            color: #66b3ff !important;
        }

        body.wcag-dark-mode .table-responsive th {
            background-color: #2a2a2a !important;
            color: #ffffff !important;
        }

        body.wcag-dark-mode .table-responsive td {
            background-color: #1a1a1a !important;
            color: #ffffff !important;
            border-color: #444 !important;
        }

        /* High Contrast Mode support for HTML widgets */
        body.wcag-high-contrast .elementor-widget-html h1,
        body.wcag-high-contrast .elementor-widget-html h2,
        body.wcag-high-contrast .elementor-widget-html h3,
        body.wcag-high-contrast .elementor-widget-html h4,
        body.wcag-high-contrast .elementor-widget-html h5,
        body.wcag-high-contrast .elementor-widget-html h6,
        body.wcag-high-contrast .elementor-widget-code h1,
        body.wcag-high-contrast .elementor-widget-code h2,
        body.wcag-high-contrast .elementor-widget-code h3,
        body.wcag-high-contrast .elementor-widget-code h4,
        body.wcag-high-contrast .elementor-widget-code h5,
        body.wcag-high-contrast .elementor-widget-code h6,
        body.wcag-high-contrast .wp-block-html h1,
        body.wcag-high-contrast .wp-block-html h2,
        body.wcag-high-contrast .wp-block-html h3,
        body.wcag-high-contrast .wp-block-html h4,
        body.wcag-high-contrast .wp-block-html h5,
        body.wcag-high-contrast .wp-block-html h6 {
            color: #ffff00 !important;
            background-color: transparent !important;
        }

        body.wcag-high-contrast .elementor-widget-html p,
        body.wcag-high-contrast .elementor-widget-html span,
        body.wcag-high-contrast .elementor-widget-html div,
        body.wcag-high-contrast .elementor-widget-html li,
        body.wcag-high-contrast .elementor-widget-code p,
        body.wcag-high-contrast .elementor-widget-code span,
        body.wcag-high-contrast .elementor-widget-code div,
        body.wcag-high-contrast .elementor-widget-code li,
        body.wcag-high-contrast .wp-block-html p,
        body.wcag-high-contrast .wp-block-html span,
        body.wcag-high-contrast .wp-block-html div,
        body.wcag-high-contrast .wp-block-html li {
            color: #ffff00 !important;
        }

        body.wcag-high-contrast .elementor-widget-html a,
        body.wcag-high-contrast .elementor-widget-code a,
        body.wcag-high-contrast .wp-block-html a {
            color: #00ffff !important;
            text-decoration: underline !important;
        }

        body.wcag-high-contrast .table-responsive th {
            background-color: #000000 !important;
            color: #ffff00 !important;
            border-color: #ffff00 !important;
        }

        body.wcag-high-contrast .table-responsive td {
            background-color: #000000 !important;
            color: #ffff00 !important;
            border-color: #ffff00 !important;
        }

        /* Classic Editor HTML blocks */
        body.wcag-dark-mode .entry-content h1,
        body.wcag-dark-mode .entry-content h2,
        body.wcag-dark-mode .entry-content h3,
        body.wcag-dark-mode .entry-content h4,
        body.wcag-dark-mode .entry-content h5,
        body.wcag-dark-mode .entry-content h6 {
            color: #ffffff !important;
        }

        body.wcag-dark-mode .entry-content p,
        body.wcag-dark-mode .entry-content span,
        body.wcag-dark-mode .entry-content div,
        body.wcag-dark-mode .entry-content li {
            color: #ffffff !important;
        }

        body.wcag-dark-mode .entry-content a {
            color: #66b3ff !important;
        }

        body.wcag-high-contrast .entry-content h1,
        body.wcag-high-contrast .entry-content h2,
        body.wcag-high-contrast .entry-content h3,
        body.wcag-high-contrast .entry-content h4,
        body.wcag-high-contrast .entry-content h5,
        body.wcag-high-contrast .entry-content h6 {
            color: #ffff00 !important;
            background-color: transparent !important;
        }

        body.wcag-high-contrast .entry-content p,
        body.wcag-high-contrast .entry-content span,
        body.wcag-high-contrast .entry-content div,
        body.wcag-high-contrast .entry-content li {
            color: #ffff00 !important;
        }

        body.wcag-high-contrast .entry-content a {
            color: #00ffff !important;
            text-decoration: underline !important;
        }
    </style>
    <?php
}
add_action( 'wp_head', 'wcag_accessible_pro_html_widget_css', 101 );

