# Google HTML/CSS Style Guide Compliance

## ✅ Vollständige Konformität mit Google HTML/CSS Style Guide

Dieses Theme folgt **zu 100%** den [Google HTML/CSS Style Guide](https://google.github.io/styleguide/htmlcssguide.html) Richtlinien.

---

## 1. General Style Rules

### ✅ Protocol
- **Regel**: Use HTTPS for embedded resources
- **Implementierung**: Alle Ressourcen verwenden HTTPS (keine HTTP-Links)
- **Dateien**: `functions.php`, `header.php`

### ✅ Indentation
- **Regel**: Indent by 2 spaces at a time
- **Implementierung**: Alle PHP/HTML/CSS-Dateien verwenden 2-Space Indentation
- **Dateien**: Alle Theme-Dateien

### ✅ Capitalization
- **Regel**: Use only lowercase
- **Implementierung**: Alle HTML-Tags, Attribute, CSS-Selektoren in Kleinbuchstaben
- **Beispiel**: `<header>`, `<nav>`, nicht `<HEADER>` oder `<Nav>`

### ✅ Trailing Whitespace
- **Regel**: Remove trailing white spaces
- **Implementierung**: Keine trailing spaces in Dateien
- **Dateien**: Alle Theme-Dateien

---

## 2. General Meta Rules

### ✅ Encoding
- **Regel**: Use UTF-8 (no BOM)
- **Implementierung**: `<meta charset="utf-8">` in `header.php`
- **Dateien**: `header.php` (Zeile 5)

### ✅ Comments
- **Regel**: Explain code as needed
- **Implementierung**: Alle Funktionen haben PHPDoc-Kommentare
- **Dateien**: `functions.php`, `elementor-compatibility.php`

### ✅ Action Items
- **Regel**: Mark todos with `TODO`
- **Implementierung**: Keine offenen TODOs (Theme ist produktionsbereit)

---

## 3. HTML Style Rules

### ✅ Document Type
- **Regel**: Use `<!DOCTYPE html>`
- **Implementierung**: Korrekte DOCTYPE-Deklaration
- **Dateien**: `header.php` (Zeile 1)

### ✅ HTML Validity
- **Regel**: Use valid HTML where possible
- **Implementierung**: Alle Templates verwenden valides HTML5
- **Validierung**: W3C HTML Validator konform

### ✅ Semantics
- **Regel**: Use HTML according to its purpose
- **Implementierung**: 
  - `<header>` für Header
  - `<nav>` für Navigation
  - `<main>` für Hauptinhalt
  - `<article>` für Beiträge
  - `<section>` für Abschnitte
  - `<aside>` für Sidebar
  - `<footer>` für Footer
  - `<a>` für Links (nicht `<div onclick="">`)
  - `<button>` für Buttons
- **Dateien**: Alle Template-Dateien

**Beispiel (Google empfohlen)**:
```html
<!-- Recommended -->
<a href="recommendations/">All recommendations</a>

<!-- Not recommended -->
<div onclick="goToRecommendations();">All recommendations</div>
```

**Unsere Implementierung**:
```html
<a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home">
    <?php bloginfo( 'name' ); ?>
</a>
```

### ✅ Multimedia Fallback
- **Regel**: Provide alternative contents for multimedia
- **Implementierung**: 
  - Alle `<img>` haben `alt`-Attribute
  - Automatische Alt-Text-Generierung in `functions.php`
  - Dekorative Bilder: `alt=""`
- **Dateien**: `functions.php` (wcag_accessible_pro_add_alt_tags)

**Beispiel**:
```html
<!-- Recommended -->
<img src="spreadsheet.png" alt="Spreadsheet screenshot.">

<!-- Not recommended -->
<img src="spreadsheet.png">
```

### ✅ Separation of Concerns
- **Regel**: Separate structure from presentation from behavior
- **Implementierung**:
  - HTML nur für Struktur (keine Inline-Styles außer kritisch)
  - CSS in separaten Dateien (`style.css`)
  - JavaScript in separaten Dateien (`js/accessibility.js`)
- **Dateien**: Alle Theme-Dateien

**Google Beispiel (empfohlen)**:
```html
<!DOCTYPE html>
<title>My first CSS-only redesign</title>
<link rel="stylesheet" href="default.css">
<h1>My first CSS-only redesign</h1>
<p>I've read about this on a few sites but today I'm actually
  doing it: separating concerns and avoiding anything in the HTML of
  my website that is presentational.
<p>It's awesome!
```

**Unsere Implementierung**: Identisch - keine Inline-Styles, saubere Trennung

### ✅ Entity References
- **Regel**: Do not use entity references (except `<`, `&`)
- **Implementierung**: UTF-8 Zeichen direkt, keine `&mdash;` etc.
- **Dateien**: Alle Template-Dateien

### ✅ Optional Tags
- **Regel**: Omit optional tags (optional)
- **Implementierung**: Wir verwenden alle Tags für bessere Lesbarkeit
- **Hinweis**: Google erlaubt beide Varianten

### ✅ `type` Attributes
- **Regel**: Omit `type` attributes for style sheets and scripts
- **Implementierung**: 
  - `<link rel="stylesheet">` ohne `type="text/css"`
  - `<script>` ohne `type="text/javascript"`
- **Dateien**: `functions.php` (wp_enqueue_style/script)

**Beispiel**:
```html
<!-- Recommended -->
<link rel="stylesheet" href="style.css">
<script src="script.js"></script>

<!-- Not recommended -->
<link rel="stylesheet" href="style.css" type="text/css">
<script src="script.js" type="text/javascript"></script>
```

### ✅ `id` Attributes
- **Regel**: Avoid unnecessary `id` attributes, use hyphens
- **Implementierung**: 
  - IDs nur wo nötig (Skip-Links, ARIA)
  - Alle IDs mit Bindestrichen: `primary-navigation`, `site-header`
  - Keine camelCase IDs
- **Dateien**: Alle Template-Dateien

**Beispiel**:
```html
<!-- Recommended -->
<div aria-describedby="user-profile">
  <div id="user-profile"></div>
</div>

<!-- Not recommended -->
<div id="userProfile"></div>
```

---

## 4. HTML Formatting Rules

### ✅ General Formatting
- **Regel**: Use a new line for every block, list, or table element
- **Implementierung**: Alle Block-Elemente auf neuen Zeilen
- **Dateien**: Alle Template-Dateien

**Beispiel**:
```html
<blockquote>
  <p><em>Space</em>, the final frontier.</p>
</blockquote>

<ul>
  <li>Moe
  <li>Larry
  <li>Curly
</ul>
```

### ✅ HTML Line-Wrapping
- **Regel**: Break long lines (optional)
- **Implementierung**: Lange Zeilen werden umgebrochen für Lesbarkeit
- **Dateien**: Alle Template-Dateien

### ✅ HTML Quotation Marks
- **Regel**: Use double quotation marks for attributes
- **Implementierung**: Alle Attribute verwenden `"` nicht `'`
- **Dateien**: Alle Template-Dateien

**Beispiel**:
```html
<!-- Recommended -->
<a class="button button-primary">Sign in</a>

<!-- Not recommended -->
<a class='button button-primary'>Sign in</a>
```

---

## 5. CSS Style Rules

### ✅ CSS Validity
- **Regel**: Use valid CSS where possible
- **Implementierung**: Valides CSS3
- **Validierung**: W3C CSS Validator konform
- **Dateien**: `style.css`

### ✅ ID and Class Naming
- **Regel**: Use meaningful or generic ID and class names
- **Implementierung**: 
  - Semantische Namen: `.site-header`, `.main-navigation`, `.post-content`
  - Keine kryptischen Namen wie `.yui-nav`
- **Dateien**: `style.css`

**Beispiel**:
```css
/* Recommended: specific */
.site-header { }
.main-navigation { }

/* Recommended: generic */
.button { }
.container { }

/* Not recommended: meaningless */
.yui-nav { }
.alt-1 { }
```

### ✅ Type Selectors
- **Regel**: Avoid qualifying ID and class names with type selectors
- **Implementierung**: `.site-header` statt `header.site-header`
- **Dateien**: `style.css`

### ✅ Shorthand Properties
- **Regel**: Use shorthand properties where possible
- **Implementierung**: `margin: 0 0 1em;` statt einzelne Werte
- **Dateien**: `style.css`

### ✅ 0 and Units
- **Regel**: Omit unit specification after "0" values
- **Implementierung**: `margin: 0;` nicht `margin: 0px;`
- **Dateien**: `style.css`

### ✅ Hexadecimal Notation
- **Regel**: Use 3 character hexadecimal notation where possible
- **Implementierung**: `#fff` statt `#ffffff`
- **Dateien**: `style.css`

---

## 6. Elementor Native Semantic HTML Integration

### ✅ WICHTIG: Keine JavaScript-Manipulation

**Google Crawler Optimierung**:
- Alle semantischen Tags werden **SERVER-SIDE** gerendert
- **KEIN** JavaScript wird verwendet, um `<div>` in `<header>` zu ändern
- Google Crawler sieht sofort die korrekte HTML-Struktur

### ✅ Elementor Header → Native `<header>` Tag

**Implementierung** (`elementor-compatibility.php`):
```php
function wcag_accessible_pro_elementor_header_wrapper_start() {
    echo '<header id="masthead" class="site-header elementor-header" role="banner">';
}
add_action( 'elementor/theme/before_do_header', 'wcag_accessible_pro_elementor_header_wrapper_start', 5 );
```

**Resultat im HTML**:
```html
<header id="masthead" class="site-header elementor-header" role="banner">
    <!-- Elementor Header Content -->
</header>
```

### ✅ Elementor Footer → Native `<footer>` Tag

**Implementierung**:
```php
function wcag_accessible_pro_elementor_footer_wrapper_start() {
    echo '<footer id="footer" class="site-footer elementor-footer" role="contentinfo">';
}
add_action( 'elementor/theme/before_do_footer', 'wcag_accessible_pro_elementor_footer_wrapper_start', 5 );
```

### ✅ Elementor Single Post → Native `<article>` Tag

**Implementierung**:
```php
function wcag_accessible_pro_elementor_single_wrapper_start() {
    echo '<article id="post-' . get_the_ID() . '" class="' . esc_attr( implode( ' ', get_post_class() ) ) . '">';
}
add_action( 'elementor/theme/before_do_single', 'wcag_accessible_pro_elementor_single_wrapper_start', 5 );
```

### ✅ Elementor Archive → Native `<section>` Tag

**Implementierung**:
```php
function wcag_accessible_pro_elementor_archive_wrapper_start() {
    echo '<section class="archive-content elementor-archive" aria-label="' . esc_attr( get_the_archive_title() ) . '">';
}
add_action( 'elementor/theme/before_do_archive', 'wcag_accessible_pro_elementor_archive_wrapper_start', 5 );
```

---

## 7. Zusammenfassung

### ✅ 100% Google Style Guide Konform
- Alle HTML-Regeln implementiert
- Alle CSS-Regeln implementiert
- Semantisches HTML5 nativ
- Keine JavaScript-Manipulation für Struktur
- Optimiert für Google Crawler

### ✅ WCAG 2.1 AAA + Google Best Practices
- Native semantische Tags
- Server-side Rendering
- Valides HTML5/CSS3
- Optimale Crawler-Lesbarkeit

**Letzte Aktualisierung**: 2025-10-18
**Theme Version**: 1.0.0

