<?php
/**
 * Breadcrumbs Widget
 *
 * @package Breadcrumbs_For_Elementor
 * @since 1.0.0
 */

namespace Breadcrumbs_For_Elementor\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Background;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Icons_Manager;
use Elementor\Repeater;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Breadcrumbs Widget Class
 *
 * @since 1.0.0
 */
class Breadcrumbs_Widget extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'breadcrumbs-for-elementor';
	}

	/**
	 * Get widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Breadcrumbs', 'breadcrumbs-for-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-navigation-horizontal';
	}

	/**
	 * Get widget categories.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( 'general' );
	}

	/**
	 * Get widget keywords.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return array( 'breadcrumbs', 'navigation', 'breadcrumb', 'crumbs', 'path' );
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->register_content_controls();
		$this->register_style_controls();
	}

	/**
	 * Register content controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_content_controls() {
		// Content Section.
		$this->start_controls_section(
			'section_content',
			array(
				'label' => esc_html__( 'Breadcrumbs Settings', 'breadcrumbs-for-elementor' ),
			)
		);

		$this->add_control(
			'breadcrumb_type',
			array(
				'label'   => esc_html__( 'Breadcrumb Type', 'breadcrumbs-for-elementor' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'auto',
				'options' => array(
					'auto'   => esc_html__( 'Auto Generate', 'breadcrumbs-for-elementor' ),
					'custom' => esc_html__( 'Custom', 'breadcrumbs-for-elementor' ),
				),
			)
		);

			$this->add_control(
				'design_preset',
				array(
					'label'       => esc_html__( 'Design Preset', 'breadcrumbs-for-elementor' ),
					'type'        => Controls_Manager::SELECT,
					'default'     => 'minimal',
					'options'     => array(
						'minimal'   => esc_html__( 'Minimal', 'breadcrumbs-for-elementor' ),
						'underline' => esc_html__( 'Underline', 'breadcrumbs-for-elementor' ),
						'pill'      => esc_html__( 'Pills', 'breadcrumbs-for-elementor' ),
						'boxed'     => esc_html__( 'Boxed', 'breadcrumbs-for-elementor' ),
						'gradient'  => esc_html__( 'Gradient', 'breadcrumbs-for-elementor' ),
						'shadow'    => esc_html__( 'Soft Shadow', 'breadcrumbs-for-elementor' ),
						'arrow'     => esc_html__( 'Arrow Trail', 'breadcrumbs-for-elementor' ),
						'dot'       => esc_html__( 'Dot Separated', 'breadcrumbs-for-elementor' ),
						'uppercase' => esc_html__( 'Uppercase', 'breadcrumbs-for-elementor' ),
						'rounded'   => esc_html__( 'Rounded', 'breadcrumbs-for-elementor' ),
					),
					'description' => esc_html__( 'Choose a pre‑designed style. You can still customize colors, typography and spacing in the Style tab.', 'breadcrumbs-for-elementor' ),
				)
			);

		$this->add_control(
			'show_home',
			array(
				'label'        => esc_html__( 'Show Home', 'breadcrumbs-for-elementor' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'breadcrumbs-for-elementor' ),
				'label_off'    => esc_html__( 'Hide', 'breadcrumbs-for-elementor' ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => array(
					'breadcrumb_type' => 'auto',
				),
			)
		);

		$this->add_control(
			'home_text',
			array(
				'label'       => esc_html__( 'Home Text', 'breadcrumbs-for-elementor' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'Home', 'breadcrumbs-for-elementor' ),
				'placeholder' => esc_html__( 'Home', 'breadcrumbs-for-elementor' ),
				'condition'   => array(
					'breadcrumb_type' => 'auto',
					'show_home'       => 'yes',
				),
			)
		);

		$this->add_control(
			'show_parent',
			array(
				'label'        => esc_html__( 'Show Parent Pages', 'breadcrumbs-for-elementor' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'breadcrumbs-for-elementor' ),
				'label_off'    => esc_html__( 'Hide', 'breadcrumbs-for-elementor' ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => array(
					'breadcrumb_type' => 'auto',
				),
			)
		);

		$this->add_control(
			'show_current',
			array(
				'label'        => esc_html__( 'Show Current Page', 'breadcrumbs-for-elementor' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'breadcrumbs-for-elementor' ),
				'label_off'    => esc_html__( 'Hide', 'breadcrumbs-for-elementor' ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => array(
					'breadcrumb_type' => 'auto',
				),
			)
		);

			// Post category controls.
			$this->add_control(
				'show_post_categories',
				array(
					'label'        => esc_html__( 'Show Post Categories', 'breadcrumbs-for-elementor' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Show', 'breadcrumbs-for-elementor' ),
					'label_off'    => esc_html__( 'Hide', 'breadcrumbs-for-elementor' ),
					'return_value' => 'yes',
					'default'      => 'yes',
					'condition'    => array(
						'breadcrumb_type' => 'auto',
					),
				)
			);

			$this->add_control(
				'include_category_hierarchy',
				array(
					'label'        => esc_html__( 'Include Subcategories', 'breadcrumbs-for-elementor' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'breadcrumbs-for-elementor' ),
					'label_off'    => esc_html__( 'No', 'breadcrumbs-for-elementor' ),
					'return_value' => 'yes',
					'default'      => 'yes',
					'condition'    => array(
						'breadcrumb_type'      => 'auto',
						'show_post_categories' => 'yes',
					),
				)
			);

		// Custom Breadcrumbs.
		$repeater = new Repeater();

		$repeater->add_control(
			'item_text',
			array(
				'label'       => esc_html__( 'Text', 'breadcrumbs-for-elementor' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'Item', 'breadcrumbs-for-elementor' ),
				'label_block' => true,
			)
		);

		$repeater->add_control(
			'item_link',
			array(
				'label'       => esc_html__( 'Link', 'breadcrumbs-for-elementor' ),
				'type'        => Controls_Manager::URL,
				'placeholder' => esc_html__( 'https://your-link.com', 'breadcrumbs-for-elementor' ),
				'default'     => array(
					'url' => '',
				),
			)
		);

		$this->add_control(
			'custom_breadcrumbs',
			array(
				'label'       => esc_html__( 'Custom Breadcrumbs', 'breadcrumbs-for-elementor' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => array(
					array(
						'item_text' => esc_html__( 'Home', 'breadcrumbs-for-elementor' ),
						'item_link' => array( 'url' => home_url( '/' ) ),
					),
					array(
						'item_text' => esc_html__( 'Page', 'breadcrumbs-for-elementor' ),
						'item_link' => array( 'url' => '' ),
					),
				),
				'title_field' => '{{{ item_text }}}',
				'condition'   => array(
					'breadcrumb_type' => 'custom',
				),
			)
		);

		$this->end_controls_section();

		// Separator Section.
		$this->start_controls_section(
			'section_separator',
			array(
				'label' => esc_html__( 'Separator', 'breadcrumbs-for-elementor' ),
			)
		);

		$this->add_control(
			'separator_type',
			array(
				'label'   => esc_html__( 'Separator Type', 'breadcrumbs-for-elementor' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'icon',
				'options' => array(
					'text' => esc_html__( 'Text', 'breadcrumbs-for-elementor' ),
					'icon' => esc_html__( 'Icon', 'breadcrumbs-for-elementor' ),
				),
			)
		);

		$this->add_control(
			'separator_text',
			array(
				'label'     => esc_html__( 'Separator Text', 'breadcrumbs-for-elementor' ),
				'type'      => Controls_Manager::TEXT,
				'default'   => '/',
				'condition' => array(
					'separator_type' => 'text',
				),
			)
		);

		$this->add_control(
			'separator_icon',
			array(
				'label'       => esc_html__( 'Separator Icon', 'breadcrumbs-for-elementor' ),
				'type'        => Controls_Manager::ICONS,
				'default'     => array(
					'value'   => 'fas fa-angle-right',
					'library' => 'fa-solid',
				),
				'recommended' => array(
					'fa-solid' => array(
						'angle-right',
						'angle-double-right',
						'chevron-right',
						'arrow-right',
						'caret-right',
					),
				),
				'condition'   => array(
					'separator_type' => 'icon',
				),
			)
		);

		$this->end_controls_section();

		// Schema Markup Section.
		$this->start_controls_section(
			'section_schema',
			array(
				'label' => esc_html__( 'Schema Markup', 'breadcrumbs-for-elementor' ),
			)
		);

		$this->add_control(
			'schema_output',
			array(
				'label'       => esc_html__( 'Schema Output', 'breadcrumbs-for-elementor' ),
				'type'        => Controls_Manager::SELECT,
				'default'     => 'json_ld',
				'options'     => array(
					'json_ld'   => esc_html__( 'JSON-LD (Recommended)', 'breadcrumbs-for-elementor' ),
					'microdata' => esc_html__( 'Microdata (HTML attributes)', 'breadcrumbs-for-elementor' ),
					'none'      => esc_html__( 'None', 'breadcrumbs-for-elementor' ),
				),
				'description' => esc_html__( 'Choose one format. JSON-LD is recommended by Google.', 'breadcrumbs-for-elementor' ),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Register style controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_style_controls() {
		// Continue in next part due to 300 line limit.
		$this->register_breadcrumb_style_controls();
		$this->register_separator_style_controls();
	}

	/**
	 * Register breadcrumb style controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_breadcrumb_style_controls() {
		$this->start_controls_section(
			'section_style_breadcrumbs',
			array(
				'label' => esc_html__( 'Breadcrumbs', 'breadcrumbs-for-elementor' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'breadcrumbs_align',
			array(
				'label'     => esc_html__( 'Alignment', 'breadcrumbs-for-elementor' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'left'   => array(
						'title' => esc_html__( 'Left', 'breadcrumbs-for-elementor' ),
						'icon'  => 'eicon-text-align-left',
					),
					'center' => array(
						'title' => esc_html__( 'Center', 'breadcrumbs-for-elementor' ),
						'icon'  => 'eicon-text-align-center',
					),
					'right'  => array(
						'title' => esc_html__( 'Right', 'breadcrumbs-for-elementor' ),
						'icon'  => 'eicon-text-align-right',
					),
				),
				'default'   => 'left',
				'selectors' => array(
					'{{WRAPPER}} .bfe-breadcrumbs' => 'text-align: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'breadcrumbs_typography',
				'selector' => '{{WRAPPER}} .bfe-breadcrumbs a, {{WRAPPER}} .bfe-breadcrumbs span',
			)
		);

		$this->start_controls_tabs( 'tabs_breadcrumbs_style' );

		$this->start_controls_tab(
			'tab_breadcrumbs_normal',
			array(
				'label' => esc_html__( 'Normal', 'breadcrumbs-for-elementor' ),
			)
		);

		$this->add_control(
			'breadcrumbs_color',
			array(
				'label'     => esc_html__( 'Color', 'breadcrumbs-for-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#666666',
				'selectors' => array(
					'{{WRAPPER}} .bfe-breadcrumbs a' => 'color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_breadcrumbs_hover',
			array(
				'label' => esc_html__( 'Hover', 'breadcrumbs-for-elementor' ),
			)
		);

		$this->add_control(
			'breadcrumbs_hover_color',
			array(
				'label'     => esc_html__( 'Color', 'breadcrumbs-for-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#2196F3',
				'selectors' => array(
					'{{WRAPPER}} .bfe-breadcrumbs a:hover' => 'color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_breadcrumbs_active',
			array(
				'label' => esc_html__( 'Active', 'breadcrumbs-for-elementor' ),
			)
		);

		$this->add_control(
			'breadcrumbs_active_color',
			array(
				'label'     => esc_html__( 'Color', 'breadcrumbs-for-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#333333',
				'selectors' => array(
					'{{WRAPPER}} .bfe-breadcrumbs .bfe-breadcrumb-current' => 'color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();
	}

	/**
	 * Register separator style controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_separator_style_controls() {
		$this->start_controls_section(
			'section_style_separator',
			array(
				'label' => esc_html__( 'Separator', 'breadcrumbs-for-elementor' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'separator_color',
			array(
				'label'     => esc_html__( 'Color', 'breadcrumbs-for-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#999999',
				'selectors' => array(
					'{{WRAPPER}} .bfe-breadcrumb-separator' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_responsive_control(
			'separator_size',
			array(
				'label'      => esc_html__( 'Size', 'breadcrumbs-for-elementor' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', 'em' ),
				'range'      => array(
					'px' => array(
						'min' => 10,
						'max' => 50,
					),
					'em' => array(
						'min' => 0.5,
						'max' => 3,
					),
				),
				'default'    => array(
					'size' => 14,
					'unit' => 'px',
				),
				'selectors'  => array(
					'{{WRAPPER}} .bfe-breadcrumb-separator' => 'font-size: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'separator_spacing',
			array(
				'label'      => esc_html__( 'Spacing', 'breadcrumbs-for-elementor' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', 'em' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 50,
					),
					'em' => array(
						'min' => 0,
						'max' => 3,
					),
				),
				'default'    => array(
					'size' => 10,
					'unit' => 'px',
				),
				'selectors'  => array(
					'{{WRAPPER}} .bfe-breadcrumb-separator' => 'margin-left: {{SIZE}}{{UNIT}}; margin-right: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Get breadcrumb trail.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @return array Breadcrumb trail.
	 */
	protected function get_breadcrumb_trail() {
		$breadcrumbs = array();
		$settings    = $this->get_settings_for_display();

		// Add home.
		if ( 'yes' === $settings['show_home'] ) {
			$breadcrumbs[] = array(
				'text' => $settings['home_text'],
				'url'  => home_url( '/' ),
			);
		}

		// Get current page.
		global $post;

		if ( is_singular() && $post ) {
			// Add parent pages.
			if ( 'yes' === $settings['show_parent'] && $post->post_parent ) {
				$parent_id = $post->post_parent;
				$parents   = array();

				while ( $parent_id ) {
					$page      = get_post( $parent_id );
					$parents[] = array(
						'text' => get_the_title( $page->ID ),
						'url'  => get_permalink( $page->ID ),
					);
					$parent_id = $page->post_parent;
				}

				$parents     = array_reverse( $parents );
				$breadcrumbs = array_merge( $breadcrumbs, $parents );
			}

			// Add post categories (with optional hierarchy).
			if ( is_singular( 'post' ) && 'yes' === $settings['show_post_categories'] ) {
				$cats = get_the_category( $post->ID );
				if ( ! empty( $cats ) && ! is_wp_error( $cats ) ) {
					$selected  = $cats[0];
					$max_depth = -1;
					foreach ( $cats as $cat ) {
						$depth = count( get_ancestors( $cat->term_id, 'category' ) );
						if ( $depth > $max_depth ) {
							$max_depth = $depth;
							$selected  = $cat;
						}
					}
					$chain = array();
					if ( 'yes' === $settings['include_category_hierarchy'] ) {
						$anc = get_ancestors( $selected->term_id, 'category' );
						$anc = array_reverse( $anc );
						foreach ( $anc as $term_id ) {
							$term = get_term( $term_id, 'category' );
							if ( $term && ! is_wp_error( $term ) ) {
								$chain[] = array(
									'text' => $term->name,
									'url'  => get_term_link( $term ),
								);
							}
						}
					}
					$chain[]     = array(
						'text' => $selected->name,
						'url'  => get_term_link( $selected ),
					);
					$breadcrumbs = array_merge( $breadcrumbs, $chain );
				}
			}

			// Add current page.
			if ( 'yes' === $settings['show_current'] ) {
				$breadcrumbs[] = array(
					'text' => get_the_title(),
					'url'  => '',
				);
			}
		} elseif ( is_archive() ) {
			if ( 'yes' === $settings['show_current'] ) {
				$breadcrumbs[] = array(
					'text' => get_the_archive_title(),
					'url'  => '',
				);
			}
		} elseif ( is_search() ) {
			if ( 'yes' === $settings['show_current'] ) {
				$breadcrumbs[] = array(
					'text' => esc_html__( 'Search Results', 'breadcrumbs-for-elementor' ),
					'url'  => '',
				);
			}
		} elseif ( is_404() ) {
			if ( 'yes' === $settings['show_current'] ) {
				$breadcrumbs[] = array(
					'text' => esc_html__( '404 Not Found', 'breadcrumbs-for-elementor' ),
					'url'  => '',
				);
			}
		}

		return $breadcrumbs;
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		// Get breadcrumbs.
		if ( 'custom' === $settings['breadcrumb_type'] ) {
			$breadcrumbs = array();
			foreach ( $settings['custom_breadcrumbs'] as $item ) {
				$breadcrumbs[] = array(
					'text' => $item['item_text'],
					'url'  => ! empty( $item['item_link']['url'] ) ? $item['item_link']['url'] : '',
				);
			}
		} else {
			$breadcrumbs = $this->get_breadcrumb_trail();
		}

		if ( empty( $breadcrumbs ) ) {
			return;
		}

		$output_mode = ! empty( $settings['schema_output'] ) ? $settings['schema_output'] : '';
		if ( empty( $output_mode ) ) {
			// Back-compat fallback for older saved settings.
			if ( isset( $settings['enable_json_ld'] ) && 'yes' === $settings['enable_json_ld'] ) {
				$output_mode = 'json_ld';
			} elseif ( ( isset( $settings['enable_microdata'] ) && 'yes' === $settings['enable_microdata'] ) || ( isset( $settings['enable_schema'] ) && 'yes' === $settings['enable_schema'] ) ) {
				$output_mode = 'microdata';
			} else {
				$output_mode = 'json_ld';
			}
		}
		$microdata_enabled = ( 'microdata' === $output_mode );
		$json_ld_enabled   = ( 'json_ld' === $output_mode );

		// Start output.
		?>
		<div class="bfe-breadcrumbs-wrapper <?php echo esc_attr( 'bfe-preset-' . sanitize_html_class( ! empty( $settings['design_preset'] ) ? $settings['design_preset'] : 'minimal' ) ); ?>">

			<nav class="bfe-breadcrumbs <?php echo esc_attr( 'bfe-preset-' . sanitize_html_class( ! empty( $settings['design_preset'] ) ? $settings['design_preset'] : 'minimal' ) ); ?>" aria-label="<?php echo esc_attr__( 'Breadcrumb', 'breadcrumbs-for-elementor' ); ?>" <?php echo $microdata_enabled ? 'itemscope itemtype="https://schema.org/BreadcrumbList"' : ''; ?> >
				<?php
				$total = count( $breadcrumbs );
				foreach ( $breadcrumbs as $index => $crumb ) :
					$position = $index + 1;
					$is_last  = $position === $total;
					?>
					<?php if ( $microdata_enabled ) : ?>
						<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">
					<?php endif; ?>

					<?php if ( ! empty( $crumb['url'] ) && ! $is_last ) : ?>
						<a href="<?php echo esc_url( $crumb['url'] ); ?>" class="bfe-breadcrumb-item" <?php echo $microdata_enabled ? 'itemprop="item"' : ''; ?>>
							<span <?php echo $microdata_enabled ? 'itemprop="name"' : ''; ?>><?php echo esc_html( $crumb['text'] ); ?></span>
						</a>
					<?php else : ?>
						<span class="bfe-breadcrumb-item bfe-breadcrumb-current" <?php echo $microdata_enabled ? 'itemprop="name"' : ''; ?> aria-current="page">
							<?php echo esc_html( $crumb['text'] ); ?>
						</span>
					<?php endif; ?>

					<?php if ( $microdata_enabled ) : ?>
						<meta itemprop="position" content="<?php echo esc_attr( $position ); ?>" />
						</span>
					<?php endif; ?>

					<?php if ( ! $is_last ) : ?>
						<span class="bfe-breadcrumb-separator">
							<?php
							if ( 'icon' === $settings['separator_type'] ) {
								// Render icon with graceful fallback to text if icon library is unavailable.
								$icon_html = '';
								if ( isset( $settings['separator_icon'] ) && is_array( $settings['separator_icon'] ) ) {
									ob_start();
									Icons_Manager::render_icon( $settings['separator_icon'], array( 'aria-hidden' => 'true' ) );
									$icon_html = trim( ob_get_clean() );
								}
								if ( '' !== $icon_html ) {
									// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
									echo $icon_html;
								} else {
									echo esc_html( ! empty( $settings['separator_text'] ) ? $settings['separator_text'] : '›' );
								}
							} else {
								echo esc_html( $settings['separator_text'] );
							}
							?>
						</span>
					<?php endif; ?>
				<?php endforeach; ?>

				<?php if ( $json_ld_enabled ) : ?>
					<script type="application/ld+json">
						<?php echo wp_json_encode( $this->generate_breadcrumb_json_ld( $breadcrumbs ), JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
					</script>
				<?php endif; ?>

			</nav>
		</div>
		<?php
	}

	/**
	 * Build BreadcrumbList JSON-LD array.
	 *
	 * @param array $breadcrumbs Breadcrumb items with text and url.
	 * @return array
	 */
	protected function generate_breadcrumb_json_ld( $breadcrumbs ) {
		$items = array();
		foreach ( $breadcrumbs as $index => $crumb ) {
			$entry = array(
				'@type'    => 'ListItem',
				'position' => $index + 1,
				'name'     => wp_strip_all_tags( (string) $crumb['text'] ),
			);
			if ( ! empty( $crumb['url'] ) ) {
				$entry['item'] = array(
					'@type' => 'WebPage',
					'@id'   => esc_url_raw( $crumb['url'] ),
				);
			}
			$items[] = $entry;
		}
		$data = array(
			'@context'        => 'https://schema.org',
			'@type'           => 'BreadcrumbList',
			'itemListElement' => $items,
		);
		/**
		 * Filter the JSON-LD breadcrumb data.
		 *
		 * @param array $data        JSON-LD data.
		 * @param array $breadcrumbs Breadcrumb items.
		 */
		return apply_filters( 'breadcrumbs_for_elementor_json_ld', $data, $breadcrumbs );
	}
}

